/**
 * Orbit - See the people who matter.
 * Privacy-first personal network intelligence
 */

const API_BASE = 'http://localhost:3001/api';

// State
let currentOnboardingStep = 1;
let currentQueryMode = null;
let hasCompletedOnboarding = localStorage.getItem('orbit_onboarding_complete') === 'true';

// DOM Elements
const elements = {
  // Onboarding
  onboarding: document.getElementById('onboarding'),
  mainApp: document.getElementById('mainApp'),
  fileInput: document.getElementById('fileInput'),
  fileDropZone: document.getElementById('fileDropZone'),
  apiKeyInput: document.getElementById('apiKeyInput'),

  // Main app
  headerStats: document.getElementById('headerStats'),
  queryInput: document.getElementById('queryInput'),
  queryBtn: document.getElementById('queryBtn'),
  queryResult: document.getElementById('queryResult'),
  quickPrompts: document.getElementById('quickPrompts'),

  // Network stats
  totalCount: document.getElementById('totalCount'),
  warmCount: document.getElementById('warmCount'),
  coolingCount: document.getElementById('coolingCount'),
  coldCount: document.getElementById('coldCount'),

  // Sections
  platformBreakdown: document.getElementById('platformBreakdown'),
  industryBreakdown: document.getElementById('industryBreakdown'),
  graphCanvas: document.getElementById('graphCanvas'),
  alerts: document.getElementById('alerts'),
  topConnections: document.getElementById('topConnections'),

  // Settings
  settingsModal: document.getElementById('settingsModal'),
  settingsApiKey: document.getElementById('settingsApiKey')
};

// Platform icons
const platformIcons = {
  linkedin: '💼',
  facebook: '📘',
  twitter: '𝕏',
  instagram: '📷',
  bluesky: '🦋',
  threads: '🧵',
  tiktok: '🎵',
  google: '📧',
  whatsapp: '💬'
};

// Industry colors
const industryColors = {
  Art: '#e879f9',
  Gaming: '#22d3ee',
  Tech: '#60a5fa',
  Education: '#4ade80',
  Finance: '#fbbf24',
  Media: '#fb923c',
  Healthcare: '#f87171',
  Consulting: '#a78bfa',
  Other: '#9ca3af'
};

// ============================================
// Onboarding
// ============================================

// Export guides for each platform
const exportGuides = {
  linkedin: {
    name: 'LinkedIn',
    steps: [
      'Go to <a href="https://www.linkedin.com/mypreferences/d/download-my-data" target="_blank">LinkedIn Data Export</a>',
      'Select "Connections" and other data you want',
      'Click "Request archive"',
      'Wait for email (can take up to 24 hours)',
      'Download the .zip file and drop it here'
    ]
  },
  facebook: {
    name: 'Facebook',
    steps: [
      'Go to <a href="https://www.facebook.com/dyi" target="_blank">Facebook Your Information</a>',
      'Click "Download Your Information"',
      'Select date range and format (JSON recommended)',
      'Select "Friends" and other categories',
      'Click "Request a Download"'
    ]
  },
  instagram: {
    name: 'Instagram',
    steps: [
      'Open Instagram app → Settings → Accounts Center',
      'Click "Your information and permissions"',
      'Select "Download your information"',
      'Choose HTML or JSON format',
      'Request and wait for email'
    ]
  },
  twitter: {
    name: 'X (Twitter)',
    steps: [
      'Go to <a href="https://twitter.com/settings/download_your_data" target="_blank">X Data Archive</a>',
      'Click "Request archive"',
      'Verify with password/2FA',
      'Wait for email notification',
      'Download and extract the .zip'
    ]
  },
  bluesky: {
    name: 'Bluesky',
    steps: [
      'Go to <a href="https://bsky.app/settings" target="_blank">Bluesky Settings</a>',
      'Scroll to "Account" section',
      'Click "Export My Data"',
      'Download the .car file',
      'Drop it here'
    ]
  },
  tiktok: {
    name: 'TikTok',
    steps: [
      'Open TikTok app → Profile → Menu (≡)',
      'Settings and privacy → Account',
      'Download your data',
      'Select data to include',
      'Request and wait for file'
    ]
  },
  threads: {
    name: 'Threads',
    steps: [
      'Open Threads app → Settings',
      'Account → Download your information',
      'Select format and content',
      'Request download',
      'Wait for email notification'
    ]
  },
  google: {
    name: 'Google Contacts',
    steps: [
      'Go to <a href="https://contacts.google.com" target="_blank">Google Contacts</a>',
      'Click "Export" in the left sidebar',
      'Choose "Google CSV" format',
      'Click "Export"',
      'Drop the .csv file here'
    ]
  }
};

// AI Provider info
const aiProviders = {
  gemini: {
    placeholder: 'AIza...',
    link: 'https://aistudio.google.com/app/apikey',
    linkText: 'Get a free Gemini key →'
  },
  openai: {
    placeholder: 'sk-...',
    link: 'https://platform.openai.com/api-keys',
    linkText: 'Get an OpenAI key →'
  },
  anthropic: {
    placeholder: 'sk-ant-...',
    link: 'https://console.anthropic.com/settings/keys',
    linkText: 'Get an Anthropic key →'
  },
  grok: {
    placeholder: 'xai-...',
    link: 'https://console.x.ai/',
    linkText: 'Get a Grok key →'
  },
  groq: {
    placeholder: 'gsk_...',
    link: 'https://console.groq.com/keys',
    linkText: 'Get a Groq key →'
  },
  openrouter: {
    placeholder: 'sk-or-...',
    link: 'https://openrouter.ai/keys',
    linkText: 'Get an OpenRouter key →'
  }
};

function showExportGuide(platform) {
  const guide = exportGuides[platform];
  if (!guide) return;

  const guideEl = document.getElementById('exportGuide');
  if (!guideEl) return;

  guideEl.innerHTML = `
    <div class="guide-content">
      <h4>📥 How to export from ${guide.name}</h4>
      <ol class="guide-steps">
        ${guide.steps.map(step => `<li>${step}</li>`).join('')}
      </ol>
    </div>
  `;
}

function updateApiKeyPlaceholder() {
  const select = document.getElementById('aiProviderSelect');
  const input = document.getElementById('apiKeyInput');
  const link = document.getElementById('getKeyLink');

  if (!select || !input) return;

  const provider = aiProviders[select.value] || aiProviders.gemini;
  input.placeholder = provider.placeholder;

  if (link) {
    link.href = provider.link;
    link.textContent = provider.linkText;
  }
}

function showOnboarding() {
  elements.onboarding.classList.remove('hidden');
  elements.mainApp.style.display = 'none';
  updateOnboardingStep(1);
}

function hideOnboarding() {
  elements.onboarding.classList.add('hidden');
  elements.mainApp.style.display = 'block';
  localStorage.setItem('orbit_onboarding_complete', 'true');
}

function updateOnboardingStep(step) {
  currentOnboardingStep = step;
  document.querySelectorAll('.onboarding-step').forEach(el => {
    el.classList.add('hidden');
  });
  const stepEl = document.querySelector(`.onboarding-step[data-step="${step}"]`);
  if (stepEl) stepEl.classList.remove('hidden');
}

function nextOnboardingStep() {
  if (currentOnboardingStep < 3) {
    updateOnboardingStep(currentOnboardingStep + 1);
  }
}

function prevOnboardingStep() {
  if (currentOnboardingStep > 1) {
    updateOnboardingStep(currentOnboardingStep - 1);
  }
}

// Handle file upload to server
async function handleFileUpload(files) {
  const dropZone = elements.fileDropZone;
  const originalContent = dropZone?.innerHTML;

  try {
    // Show upload progress
    if (dropZone) {
      dropZone.innerHTML = `
        <div class="upload-progress">
          <div class="spinner"></div>
          <p>Uploading ${files.length} file(s)...</p>
        </div>
      `;
    }

    // Create FormData and append files
    const formData = new FormData();
    for (let i = 0; i < files.length; i++) {
      formData.append('files', files[i]);
    }

    // Upload to server
    const response = await fetch(`${API_BASE}/upload`, {
      method: 'POST',
      body: formData
    });

    const result = await response.json();

    if (!response.ok) {
      throw new Error(result.error || 'Upload failed');
    }

    // Show success message
    if (dropZone) {
      dropZone.innerHTML = `
        <div class="upload-success">
          <div class="success-icon">✅</div>
          <p><strong>Success!</strong></p>
          <p>${result.message}</p>
          <p class="upload-hint">Drop more files or continue to the next step</p>
        </div>
      `;
    }

    console.log('Upload successful:', result);

  } catch (error) {
    console.error('Upload error:', error);

    // Show error message
    if (dropZone) {
      dropZone.innerHTML = `
        <div class="upload-error">
          <div class="error-icon">❌</div>
          <p><strong>Upload failed</strong></p>
          <p>${error.message}</p>
          <button onclick="location.reload()" class="btn-secondary">Try Again</button>
        </div>
      `;
    }
  }
}

function completeOnboarding() {
  // Save AI provider selection
  const providerSelect = document.getElementById('aiProviderSelect');
  if (providerSelect) {
    localStorage.setItem('orbit_ai_provider', providerSelect.value);
  }

  // Save API key if provided
  const apiKey = elements.apiKeyInput?.value?.trim();
  if (apiKey) {
    localStorage.setItem('orbit_api_key', apiKey);
  }

  hideOnboarding();
  init();
}

// ============================================
// Settings
// ============================================

function openSettings() {
  elements.settingsModal.classList.remove('hidden');

  // Load saved provider
  const savedProvider = localStorage.getItem('orbit_ai_provider') || 'gemini';
  const providerSelect = document.getElementById('settingsProvider');
  if (providerSelect) {
    providerSelect.value = savedProvider;
  }

  // Load saved key
  const savedKey = localStorage.getItem('orbit_api_key') || '';
  if (elements.settingsApiKey) {
    elements.settingsApiKey.value = savedKey;
  }

  // Update placeholder based on provider
  updateSettingsPlaceholder();

  // Load data sources status
  loadDataSources();
}

// Load and display data sources status in settings
async function loadDataSources() {
  const container = document.getElementById('dataSources');
  if (!container) return;

  const allPlatforms = [
    { id: 'linkedin', name: 'LinkedIn', icon: '💼', guide: 'Settings → Data Privacy → Get a copy of your data', url: 'https://www.linkedin.com/mypreferences/d/download-my-data' },
    { id: 'facebook', name: 'Facebook', icon: '📘', guide: 'Settings → Your Information → Download Your Information', url: 'https://www.facebook.com/dyi' },
    { id: 'instagram', name: 'Instagram', icon: '📸', guide: 'Settings → Your Activity → Download Your Information', url: 'https://www.instagram.com/download/request' },
    { id: 'twitter', name: 'X (Twitter)', icon: '𝕏', guide: 'Settings → Your Account → Download an archive', url: 'https://twitter.com/settings/download_your_data' },
    { id: 'bluesky', name: 'Bluesky', icon: '🦋', guide: 'Settings → Advanced → Export my data', url: 'https://bsky.app/settings' },
    { id: 'threads', name: 'Threads', icon: '🧵', guide: 'Via Instagram data export (includes Threads)', url: 'https://www.instagram.com/download/request' },
    { id: 'google', name: 'Google', icon: '📧', guide: 'Go to contacts.google.com → Export → Choose CSV', url: 'https://contacts.google.com' }
  ];

  // Store platforms globally for guide lookup
  window.platformGuides = allPlatforms;

  try {
    const response = await fetch(`${API_BASE}/overview`);
    const data = await response.json();
    const breakdown = data.platformBreakdown || {};

    container.innerHTML = allPlatforms.map(platform => {
      const count = breakdown[platform.id] || 0;
      const isConnected = count > 0;

      return `
        <div class="data-source-card ${isConnected ? 'connected' : ''}">
          <div class="platform-icon">${platform.icon}</div>
          <div class="platform-name">${platform.name}</div>
          <div class="contact-count ${isConnected ? 'connected' : ''}">
            ${isConnected ? `${count} contacts` : 'Not imported'}
          </div>
          <span class="status-badge ${isConnected ? 'connected' : 'not-connected'}">
            ${isConnected ? '✓ Connected' : 'Add'}
          </span>
          ${!isConnected ? `<a class="import-link" href="${platform.url}" target="_blank" title="${platform.guide}">How to export ↗</a>` : ''}
        </div>
      `;
    }).join('');
  } catch (error) {
    container.innerHTML = '<div class="loading-sources">Could not load platform data</div>';
  }
}

function showPlatformGuide(platformId) {
  const platform = window.platformGuides?.find(p => p.id === platformId);
  if (platform) {
    window.open(platform.url, '_blank');
  }
}

function closeSettings() {
  elements.settingsModal.classList.add('hidden');
}

function updateSettingsPlaceholder() {
  const select = document.getElementById('settingsProvider');
  const input = document.getElementById('settingsApiKey');

  if (!select || !input) return;

  const provider = aiProviders[select.value] || aiProviders.gemini;
  input.placeholder = provider.placeholder;
}

function saveApiKey() {
  const providerSelect = document.getElementById('settingsProvider');
  const key = elements.settingsApiKey?.value?.trim();

  // Save provider
  if (providerSelect) {
    localStorage.setItem('orbit_ai_provider', providerSelect.value);
  }

  // Save key
  if (key) {
    localStorage.setItem('orbit_api_key', key);
  }

  alert('Settings saved!');
}

function reimportData() {
  closeSettings();
  showOnboarding();
  updateOnboardingStep(2);
}

function clearAllData() {
  if (confirm('Are you sure you want to clear all data? This cannot be undone.')) {
    localStorage.removeItem('orbit_onboarding_complete');
    localStorage.removeItem('orbit_api_key');
    localStorage.removeItem('orbit_ai_provider');
    location.reload();
  }
}

// ============================================
// Query Functions
// ============================================

function setQuery(query) {
  elements.queryInput.value = query;
  elements.queryInput.focus();
}

// ============================================
// Initialization
// ============================================

async function init() {
  // ALWAYS setup event listeners first (so drop zone works in onboarding)
  setupEventListeners();

  // Check if data exists
  try {
    const response = await fetch(`${API_BASE}/overview`);
    const data = await response.json();

    if (data.totalContacts > 0) {
      // Data exists, load dashboard
      await Promise.all([
        loadOverview(),
        loadAlerts(),
        loadTopConnections(),
        loadIndustries(),
        loadNetworkGraph()
      ]);
      updateQuickPrompts(true);
    } else {
      // No data - but only show onboarding if they haven't completed it
      // This allows users to skip onboarding and see empty dashboard
      if (!hasCompletedOnboarding) {
        showOnboarding();
      } else {
        // Show empty dashboard with helpful message
        updateQuickPrompts(false);
        showEmptyDashboardMessage();
      }
    }
  } catch (error) {
    console.error('Failed to initialize:', error);
    // Server might be offline - show helpful message
    showServerOfflineMessage();
  }
}

function showEmptyDashboardMessage() {
  const headerStats = document.getElementById('headerStats');
  if (headerStats) {
    headerStats.innerHTML = `
      <span class="stat-badge warning">No data yet</span>
      <button onclick="reimportData()" class="btn-small">Import Data</button>
    `;
  }

  // Update stats to show zeros
  if (elements.totalCount) elements.totalCount.textContent = '0';
  if (elements.warmCount) elements.warmCount.textContent = '0';
  if (elements.coolingCount) elements.coolingCount.textContent = '0';
  if (elements.coldCount) elements.coldCount.textContent = '0';
}

function showServerOfflineMessage() {
  const mainApp = document.getElementById('mainApp');
  if (mainApp) {
    mainApp.innerHTML = `
      <div class="empty-state-container">
        <div class="empty-state-icon">🔌</div>
        <h2>Server Not Running</h2>
        <p>Make sure the Orbit server is running:</p>
        <code style="display: block; background: #1e293b; padding: 1rem; border-radius: 8px; margin: 1rem 0;">npm run dev</code>
        <button onclick="location.reload()" class="btn-primary">Retry Connection</button>
      </div>
    `;
  }
}

async function updateQuickPrompts(hasData) {
  if (!elements.quickPrompts) return;

  if (hasData) {
    try {
      // Fetch personalized prompts based on user's data
      const response = await fetch(`${API_BASE}/smart-prompts`);
      const data = await response.json();

      if (data.hasData && data.prompts.length > 0) {
        const promptButtons = data.prompts
          .map(p => `<button class="prompt-btn" onclick="setQuery('${p.text.replace(/'/g, "\\'")}')">${p.text}</button>`)
          .join('\n');

        elements.quickPrompts.innerHTML = `
          <p class="prompts-label">Try asking:</p>
          ${promptButtons}
        `;
      } else {
        // Fallback prompts
        elements.quickPrompts.innerHTML = `
          <p class="prompts-label">Try asking:</p>
          <button class="prompt-btn" onclick="setQuery('Who should I reconnect with?')">Who should I reconnect with?</button>
          <button class="prompt-btn" onclick="setQuery('Who can help me reach my goals?')">Who can help me reach my goals?</button>
          <button class="prompt-btn" onclick="setQuery('Who owes me a response?')">Who owes me a response?</button>
        `;
      }
    } catch (error) {
      console.error('Failed to fetch smart prompts:', error);
      // Fallback on error
      elements.quickPrompts.innerHTML = `
        <p class="prompts-label">Try asking:</p>
        <button class="prompt-btn" onclick="setQuery('Who should I reconnect with?')">Who should I reconnect with?</button>
        <button class="prompt-btn" onclick="setQuery('Who can help me reach my goals?')">Who can help me reach my goals?</button>
      `;
    }
  } else {
    // Generic prompts for new users
    elements.quickPrompts.innerHTML = `
      <p class="prompts-label">Import your data to ask questions like:</p>
      <button class="prompt-btn" disabled>Who can introduce me to [industry]?</button>
      <button class="prompt-btn" disabled>Who should I reconnect with?</button>
      <button class="prompt-btn" disabled>Who knows people at [company]?</button>
    `;
  }
}

// Event listeners
function setupEventListeners() {
  if (elements.queryBtn) {
    elements.queryBtn.addEventListener('click', handleQuery);
  }
  if (elements.queryInput) {
    elements.queryInput.addEventListener('keypress', (e) => {
      if (e.key === 'Enter') handleQuery();
    });
  }

  // File drop zone
  if (elements.fileDropZone) {
    elements.fileDropZone.addEventListener('dragover', (e) => {
      e.preventDefault();
      elements.fileDropZone.classList.add('drag-over');
    });

    elements.fileDropZone.addEventListener('dragleave', () => {
      elements.fileDropZone.classList.remove('drag-over');
    });

    elements.fileDropZone.addEventListener('drop', async (e) => {
      e.preventDefault();
      elements.fileDropZone.classList.remove('drag-over');

      const files = e.dataTransfer.files;
      if (files.length === 0) return;

      await handleFileUpload(files);
    });

    // Also handle click to select files
    elements.fileDropZone.addEventListener('click', () => {
      elements.fileInput?.click();
    });
  }

  // Handle file input change
  if (elements.fileInput) {
    elements.fileInput.addEventListener('change', async (e) => {
      const files = e.target.files;
      if (files.length > 0) {
        await handleFileUpload(files);
      }
    });
  }
}


// Load network overview
async function loadOverview() {
  try {
    const response = await fetch(`${API_BASE}/overview`);
    const data = await response.json();

    if (elements.headerStats) {
      elements.headerStats.innerHTML = `
        <span class="stat-badge">${data.totalContacts} contacts</span>
        <span class="stat-badge">${data.platforms?.length || 0} platforms</span>
      `;
    }

    if (data.relationshipHealth) {
      const health = data.relationshipHealth;

      // Update network stats boxes
      if (elements.totalCount) elements.totalCount.textContent = health.total || 0;
      if (elements.warmCount) elements.warmCount.textContent = health.warm || 0;
      if (elements.coolingCount) elements.coolingCount.textContent = health.cooling || 0;
      if (elements.coldCount) elements.coldCount.textContent = health.cold || 0;
    }

    if (data.platformBreakdown && Object.keys(data.platformBreakdown).length > 0 && elements.platformBreakdown) {
      const maxCount = Math.max(...Object.values(data.platformBreakdown));

      elements.platformBreakdown.innerHTML = Object.entries(data.platformBreakdown)
        .sort((a, b) => b[1] - a[1])
        .map(([platform, count]) => `
          <div class="platform-item">
            <span class="platform-icon">${platformIcons[platform] || '📱'}</span>
            <div class="platform-bar-wrapper">
              <div class="platform-name">${capitalize(platform)}</div>
              <div class="platform-bar">
                <div class="platform-bar-fill" style="width: ${(count / maxCount) * 100}%"></div>
              </div>
            </div>
            <span class="platform-count">${count}</span>
          </div>
        `).join('');
    }
  } catch (error) {
    console.error('Failed to load overview:', error);
    if (elements.headerStats) {
      elements.headerStats.innerHTML = `<span class="stat-badge">Server offline</span>`;
    }
  }
}

// Load industries breakdown
async function loadIndustries() {
  try {
    const response = await fetch(`${API_BASE}/industries`);
    const industries = await response.json();

    if (Object.keys(industries).length === 0) {
      elements.industryBreakdown.innerHTML = '<p class="empty-state">No industry data</p>';
      return;
    }

    const sorted = Object.entries(industries)
      .sort((a, b) => b[1].count - a[1].count)
      .slice(0, 6);

    const maxCount = sorted[0][1].count;

    elements.industryBreakdown.innerHTML = sorted.map(([industry, data]) => `
            <div class="industry-item">
                <div class="industry-header">
                    <span class="industry-dot" style="background: ${industryColors[industry] || '#9ca3af'}"></span>
                    <span class="industry-name">${industry}</span>
                    <span class="industry-count">${data.count}</span>
                </div>
                <div class="industry-bar">
                    <div class="industry-bar-fill" style="width: ${(data.count / maxCount) * 100}%; background: ${industryColors[industry] || '#9ca3af'}"></div>
                </div>
            </div>
        `).join('');
  } catch (error) {
    console.error('Failed to load industries:', error);
  }
}

// Load weekly digest
async function loadDigest() {
  try {
    const response = await fetch(`${API_BASE}/digest`);
    const digest = await response.json();

    if (!digest.suggestions || digest.suggestions.length === 0) {
      elements.weeklyDigest.innerHTML = '<p class="empty-state">No suggestions this week 🎉</p>';
      return;
    }

    elements.weeklyDigest.innerHTML = digest.suggestions.map(s => `
            <div class="digest-item ${s.type}">
                <div class="digest-icon">${s.type === 'reconnect' ? '🔄' : '🤝'}</div>
                <div class="digest-content">
                    <div class="digest-contact">${s.contact.name}</div>
                    <div class="digest-action">${s.action}</div>
                </div>
            </div>
        `).join('');
  } catch (error) {
    console.error('Failed to load digest:', error);
  }
}

// Load company intelligence
async function loadCompanyIntel() {
  if (!elements.companyIntel) return;

  try {
    const response = await fetch(`${API_BASE}/company-intel`);
    const data = await response.json();

    if (!data.companies || data.companies.length === 0) {
      elements.companyIntel.innerHTML = `<p class="empty-state">No connections at followed companies yet</p>`;
      return;
    }

    elements.companyIntel.innerHTML = `
      <div class="company-stats">
        <span class="stat-highlight">${data.withConnections}</span> of ${data.totalFollowed} followed companies have connections
      </div>
      ${data.companies.slice(0, 5).map(c => `
        <div class="company-item">
          <div class="company-header">
            <span class="company-name">${c.company}</span>
            <span class="company-count">${c.contactCount} contact${c.contactCount > 1 ? 's' : ''}</span>
          </div>
          <div class="company-contacts">
            ${c.contacts.map(contact => `
              <span class="company-contact" title="${contact.title || ''}">${contact.name.split(' ')[0]}</span>
            `).join('')}
          </div>
        </div>
      `).join('')}
    `;
  } catch (error) {
    console.error('Failed to load company intel:', error);
  }
}

// Clear query and results
function clearQuery() {
  elements.queryInput.value = '';
  elements.queryResult.classList.add('hidden');
  elements.queryResult.innerHTML = '';
  elements.queryInput.placeholder = currentQueryMode
    ? (currentQueryMode === 'art' ? '🎨 Avery Lake mode: Ask about galleries, curators, Art Basel...' : '💼 Conexkt mode: Ask about education, tech, consulting...')
    : 'Ask anything about your network...';
}

// Load network graph
async function loadNetworkGraph() {
  try {
    const response = await fetch(`${API_BASE}/graph`);
    const graph = await response.json();

    if (!graph.nodes || graph.nodes.length === 0) {
      return;
    }

    drawGraph(graph);
  } catch (error) {
    console.error('Failed to load graph:', error);
  }
}

// Draw network graph on canvas with radial industry layout
let graphAnimation = null;
let graphNodes = [];
let selectedNode = null;

function drawGraph(graph) {
  const canvas = elements.graphCanvas;
  if (!canvas) return;

  const ctx = canvas.getContext('2d');
  const width = canvas.width;
  const height = canvas.height;
  const centerX = width / 2;
  const centerY = height / 2;

  // Stop previous animation
  if (graphAnimation) {
    cancelAnimationFrame(graphAnimation);
  }

  // Group nodes by industry
  const industryGroups = {};
  const industries = [];

  for (const node of graph.nodes) {
    const industry = node.industry || 'Other';
    if (!industryGroups[industry]) {
      industryGroups[industry] = [];
      industries.push(industry);
    }
    industryGroups[industry].push(node);
  }

  // Sort industries by size for better layout
  industries.sort((a, b) => industryGroups[b].length - industryGroups[a].length);

  // Position nodes in radial sectors by industry
  const sectorAngle = (Math.PI * 2) / industries.length;
  const innerRadius = 60;
  const outerRadius = Math.min(width, height) / 2 - 40;

  graphNodes = [];

  industries.forEach((industry, sectorIndex) => {
    const nodes = industryGroups[industry];
    const baseAngle = sectorIndex * sectorAngle - Math.PI / 2;
    const sectorWidth = sectorAngle * 0.8;

    nodes.forEach((node, nodeIndex) => {
      // Spread nodes within sector
      const angleOffset = (nodeIndex / Math.max(1, nodes.length - 1) - 0.5) * sectorWidth;
      const angle = baseAngle + angleOffset;

      // Vary radius based on warmth (warmer = closer to center)
      const warmthFactor = node.warmth / 100;
      const radius = outerRadius - (warmthFactor * (outerRadius - innerRadius) * 0.6);

      graphNodes.push({
        ...node,
        x: centerX + Math.cos(angle) * radius,
        y: centerY + Math.sin(angle) * radius,
        targetX: centerX + Math.cos(angle) * radius,
        targetY: centerY + Math.sin(angle) * radius,
        sectorAngle: baseAngle,
        industry: industry
      });
    });
  });

  // Mouse interaction
  let hoveredNode = null;

  canvas.onmousemove = (e) => {
    const rect = canvas.getBoundingClientRect();
    const mx = (e.clientX - rect.left) * (canvas.width / rect.width);
    const my = (e.clientY - rect.top) * (canvas.height / rect.height);

    hoveredNode = graphNodes.find(n => {
      const dx = n.x - mx;
      const dy = n.y - my;
      return Math.sqrt(dx * dx + dy * dy) < 12;
    });
    canvas.style.cursor = hoveredNode ? 'pointer' : 'default';
  };

  canvas.onclick = (e) => {
    if (hoveredNode) {
      selectedNode = selectedNode === hoveredNode ? null : hoveredNode;
    } else {
      selectedNode = null;
    }
  };

  canvas.onmouseleave = () => {
    hoveredNode = null;
  };

  // Animation loop
  function animate() {
    // Gentle floating animation
    const time = Date.now() / 2000;

    // Draw background
    ctx.fillStyle = '#0a0a0f';
    ctx.fillRect(0, 0, width, height);

    // Draw sector backgrounds and labels
    industries.forEach((industry, i) => {
      const startAngle = i * sectorAngle - Math.PI / 2 - sectorAngle / 2;
      const endAngle = startAngle + sectorAngle;
      const color = industryColors[industry] || '#9ca3af';

      // Sector wedge
      ctx.beginPath();
      ctx.moveTo(centerX, centerY);
      ctx.arc(centerX, centerY, outerRadius + 10, startAngle, endAngle);
      ctx.closePath();
      ctx.fillStyle = color + '15';
      ctx.fill();

      // Sector label
      const labelAngle = startAngle + sectorAngle / 2;
      const labelRadius = outerRadius + 25;
      const labelX = centerX + Math.cos(labelAngle) * labelRadius;
      const labelY = centerY + Math.sin(labelAngle) * labelRadius;

      ctx.save();
      ctx.translate(labelX, labelY);
      ctx.fillStyle = color;
      ctx.font = 'bold 10px Inter, sans-serif';
      ctx.textAlign = 'center';
      ctx.textBaseline = 'middle';
      ctx.fillText(`${industry} (${industryGroups[industry].length})`, 0, 0);
      ctx.restore();
    });

    // Draw center circle
    ctx.beginPath();
    ctx.arc(centerX, centerY, innerRadius - 10, 0, Math.PI * 2);
    ctx.fillStyle = 'rgba(124, 58, 237, 0.1)';
    ctx.fill();
    ctx.strokeStyle = 'rgba(124, 58, 237, 0.3)';
    ctx.lineWidth = 1;
    ctx.stroke();

    // Center label
    ctx.fillStyle = '#f0f0f5';
    ctx.font = 'bold 11px Inter, sans-serif';
    ctx.textAlign = 'center';
    ctx.fillText('YOU', centerX, centerY);

    // Draw nodes with gentle floating
    for (const node of graphNodes) {
      const floatX = Math.sin(time + node.id) * 2;
      const floatY = Math.cos(time * 0.7 + node.id * 0.5) * 2;
      node.x = node.targetX + floatX;
      node.y = node.targetY + floatY;

      const color = industryColors[node.industry] || '#9ca3af';
      const size = Math.max(5, Math.min(12, node.warmth / 8));
      const isHovered = node === hoveredNode;
      const isSelected = node === selectedNode;

      // Glow effect
      if (isHovered || isSelected) {
        ctx.beginPath();
        ctx.arc(node.x, node.y, size + 8, 0, Math.PI * 2);
        ctx.fillStyle = color + '60';
        ctx.fill();
      }

      // Node circle
      ctx.beginPath();
      ctx.arc(node.x, node.y, size, 0, Math.PI * 2);
      ctx.fillStyle = isSelected ? '#ffffff' : color;
      ctx.fill();

      // Border for selected
      if (isSelected) {
        ctx.strokeStyle = color;
        ctx.lineWidth = 2;
        ctx.stroke();
      }

      // Label for hovered/selected/high warmth
      if (isHovered || isSelected || node.warmth > 70) {
        ctx.fillStyle = '#f0f0f5';
        ctx.font = (isHovered || isSelected) ? 'bold 10px Inter, sans-serif' : '9px Inter, sans-serif';
        ctx.textAlign = 'center';
        ctx.fillText(node.name.split(' ')[0], node.x, node.y + size + 10);
      }
    }

    // Draw info panel for selected node
    if (selectedNode) {
      const panelWidth = 180;
      const panelHeight = 80;
      const panelX = 10;
      const panelY = height - panelHeight - 10;

      // Panel background
      ctx.fillStyle = 'rgba(15, 15, 25, 0.95)';
      ctx.strokeStyle = industryColors[selectedNode.industry] || '#7c3aed';
      ctx.lineWidth = 2;
      ctx.beginPath();
      ctx.roundRect(panelX, panelY, panelWidth, panelHeight, 8);
      ctx.fill();
      ctx.stroke();

      // Panel content
      ctx.fillStyle = '#ffffff';
      ctx.font = 'bold 12px Inter, sans-serif';
      ctx.textAlign = 'left';
      ctx.fillText(selectedNode.name, panelX + 10, panelY + 22);

      ctx.fillStyle = '#9ca3af';
      ctx.font = '10px Inter, sans-serif';
      ctx.fillText(selectedNode.industry || 'Unknown', panelX + 10, panelY + 40);

      ctx.fillStyle = industryColors[selectedNode.industry] || '#7c3aed';
      ctx.font = 'bold 11px Inter, sans-serif';
      ctx.fillText(`Warmth: ${Math.round(selectedNode.warmth)}`, panelX + 10, panelY + 58);
    }

    graphAnimation = requestAnimationFrame(animate);
  }

  animate();
}


// Load alerts
async function loadAlerts() {
  try {
    const response = await fetch(`${API_BASE}/alerts`);
    const alerts = await response.json();

    if (alerts.length === 0) {
      elements.alerts.innerHTML = '<p class="empty-state">No alerts - your network is healthy! 🎉</p>';
      return;
    }

    // Icons for different alert types
    const alertIcons = {
      cooling: '🔥',
      reciprocity: '🤝',
      opportunity: '✨',
      unfinished: '💬'
    };

    elements.alerts.innerHTML = alerts.slice(0, 6).map(alert => `
            <div class="alert-item ${alert.priority} ${alert.type}">
                <div class="alert-icon">${alertIcons[alert.type] || '⚠️'}</div>
                <div class="alert-content">
                    <div class="alert-contact">${alert.contact}</div>
                    <div class="alert-message">${alert.message}</div>
                    <div class="alert-suggestion">${alert.suggestion}</div>
                </div>
            </div>
        `).join('');
  } catch (error) {
    console.error('Failed to load alerts:', error);
  }
}


// Load top connections
async function loadTopConnections() {
  try {
    const response = await fetch(`${API_BASE}/contacts`);
    const contacts = await response.json();

    if (contacts.length === 0) {
      elements.topConnections.innerHTML = '<p class="empty-state">Add your data exports to see connections</p>';
      return;
    }

    elements.topConnections.innerHTML = contacts.slice(0, 8).map(contact => {
      // Generate platform links
      const platformLinks = generatePlatformLinks(contact);

      return `
        <div class="connection-item">
          <div class="connection-avatar">${getInitials(contact.name)}</div>
          <div class="connection-info">
            <div class="connection-name">${contact.name}</div>
            <div class="connection-role">${contact.title || ''} ${contact.company ? `at ${contact.company}` : ''}</div>
            ${platformLinks ? `<div class="connection-platforms">${platformLinks}</div>` : ''}
          </div>
          <div class="connection-score">${contact.scores?.warmth || 0}</div>
        </div>
      `;
    }).join('');
  } catch (error) {
    console.error('Failed to load connections:', error);
  }
}

// Generate clickable platform links for a contact
function generatePlatformLinks(contact) {
  const links = [];

  // LinkedIn profile URL
  if (contact.platforms?.includes('linkedin') || contact.linkedinUrl) {
    const url = contact.linkedinUrl || `https://www.linkedin.com/search/results/all/?keywords=${encodeURIComponent(contact.name)}`;
    links.push(`<a href="${url}" target="_blank" class="platform-link linkedin" title="LinkedIn">💼</a>`);
  }

  // Bluesky profile
  if (contact.platforms?.includes('bluesky') || contact.blueskyHandle) {
    const handle = contact.blueskyHandle || contact.name.toLowerCase().replace(/\s+/g, '');
    links.push(`<a href="https://bsky.app/profile/${handle}" target="_blank" class="platform-link bluesky" title="Bluesky">🦋</a>`);
  }

  // Twitter/X profile
  if (contact.platforms?.includes('twitter') || contact.twitterHandle) {
    const handle = contact.twitterHandle || '';
    if (handle) {
      links.push(`<a href="https://x.com/${handle}" target="_blank" class="platform-link twitter" title="X">𝕏</a>`);
    }
  }

  // Facebook profile
  if (contact.platforms?.includes('facebook') || contact.facebookUrl) {
    const url = contact.facebookUrl || `https://www.facebook.com/search/top/?q=${encodeURIComponent(contact.name)}`;
    links.push(`<a href="${url}" target="_blank" class="platform-link facebook" title="Facebook">👥</a>`);
  }

  // Instagram profile
  if (contact.platforms?.includes('instagram') || contact.instagramHandle) {
    const handle = contact.instagramHandle || '';
    if (handle) {
      links.push(`<a href="https://instagram.com/${handle}" target="_blank" class="platform-link instagram" title="Instagram">📷</a>`);
    }
  }

  return links.join('');
}

// (currentQueryMode is defined at the top)

// Handle query with goal-specific mode
async function handleQuery() {
  const question = elements.queryInput.value.trim();
  if (!question) return;

  elements.queryBtn.disabled = true;
  elements.queryBtn.innerHTML = '<span class="btn-text">Thinking...</span><span class="btn-icon">⏳</span>';
  elements.queryResult.classList.remove('hidden');
  elements.queryResult.innerHTML = '<p class="empty-state">Analyzing your network...</p>';

  try {
    // Use goal-specific endpoint if mode selected
    const endpoint = currentQueryMode ? 'query-goal' : 'query';
    const body = currentQueryMode
      ? { question, mode: currentQueryMode }
      : { question };

    const response = await fetch(`${API_BASE}/${endpoint}`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(body)
    });

    const result = await response.json();
    const modeBadge = currentQueryMode
      ? `<span class="mode-badge ${currentQueryMode}">${currentQueryMode === 'art' ? '🎨 Avery Lake' : '💼 Conexkt'}</span>`
      : '';

    elements.queryResult.innerHTML = `
      <h4>Answer ${modeBadge}<span class="source-badge">${result.source || 'local'}</span></h4>
      <p>${result.answer}</p>
      ${result.recommendations?.length > 0 ? `
        <h4 style="margin-top: 1rem;">Recommended Contacts</h4>
        <div class="connections-list">
          ${result.recommendations.map(rec => {
      const platformLinks = generatePlatformLinks(rec);
      return `
              <div class="connection-item clickable">
                <div class="connection-avatar">${getInitials(rec.name)}</div>
                <div class="connection-info">
                  <div class="connection-name">${rec.name}</div>
                  <div class="connection-role">${rec.title || ''} ${rec.company ? `at ${rec.company}` : ''}</div>
                  ${platformLinks ? `<div class="connection-platforms">${platformLinks}</div>` : ''}
                  ${rec.industry ? `<span class="industry-tag" style="background: ${industryColors[rec.industry] || '#94a3b8'}">${rec.industry}</span>` : ''}
                </div>
              </div>
            `;
    }).join('')}
        </div>
      ` : ''}
        `;
  } catch (error) {
    elements.queryResult.innerHTML = `<p>Error: Could not process query. Is the server running?</p>`;
  } finally {
    elements.queryBtn.disabled = false;
    elements.queryBtn.innerHTML = '<span class="btn-text">Ask</span><span class="btn-icon">→</span>';
  }
}

// Set query mode
function setQueryMode(mode) {
  currentQueryMode = currentQueryMode === mode ? null : mode;

  // Update button states
  document.querySelectorAll('.mode-btn').forEach(btn => {
    btn.classList.remove('active');
    if (btn.dataset.mode === currentQueryMode) {
      btn.classList.add('active');
    }
  });

  // Update placeholder
  if (currentQueryMode === 'art') {
    elements.queryInput.placeholder = '🎨 Avery Lake mode: Ask about galleries, curators, Art Basel...';
  } else if (currentQueryMode === 'business') {
    elements.queryInput.placeholder = '💼 Conexkt mode: Ask about education, tech, consulting...';
  } else {
    elements.queryInput.placeholder = 'Ask anything about your network...';
  }
}

// Handle warm path finder
async function handlePathFinder() {
  const target = elements.pathInput.value.trim();
  if (!target) return;

  elements.pathBtn.disabled = true;
  elements.pathBtn.textContent = 'Searching...';
  elements.pathResults.classList.remove('hidden');
  elements.pathResults.innerHTML = '<p class="empty-state">Finding paths...</p>';

  try {
    const response = await fetch(`${API_BASE}/warm-path`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ target })
    });

    const result = await response.json();

    if (result.paths.length === 0) {
      elements.pathResults.innerHTML = '<p class="empty-state">No paths found. Try a different target.</p>';
      return;
    }

    elements.pathResults.innerHTML = result.paths.map((path, i) => `
            <div class="path-card">
                <div class="path-rank">#${i + 1}</div>
                <div class="path-info">
                    <div class="path-name">${path.contact.name}</div>
                    <div class="path-role">${path.contact.title || ''} ${path.contact.company ? `at ${path.contact.company}` : ''}</div>
                    <div class="path-approach">${path.suggestedApproach}</div>
                </div>
                <div class="path-scores">
                    <span class="score-badge relevance">Relevance: ${path.relevanceScore}</span>
                    <span class="score-badge warmth">Warmth: ${path.warmthScore}</span>
                </div>
            </div>
        `).join('');
  } catch (error) {
    elements.pathResults.innerHTML = '<p class="empty-state">Error finding paths. Is the server running?</p>';
  } finally {
    elements.pathBtn.disabled = false;
    elements.pathBtn.textContent = 'Find Paths';
  }
}

// Utility functions
function capitalize(str) {
  return str.charAt(0).toUpperCase() + str.slice(1);
}

function getInitials(name) {
  if (!name) return '?';
  return name.split(' ').map(n => n[0]).join('').toUpperCase().slice(0, 2);
}

// Expose functions for HTML onclick handlers
window.setQueryMode = setQueryMode;
window.clearQuery = clearQuery;
window.setQuery = setQuery;
window.nextOnboardingStep = nextOnboardingStep;
window.prevOnboardingStep = prevOnboardingStep;
window.completeOnboarding = completeOnboarding;
window.openSettings = openSettings;
window.closeSettings = closeSettings;
window.saveApiKey = saveApiKey;
window.reimportData = reimportData;
window.clearAllData = clearAllData;
window.showExportGuide = showExportGuide;
window.updateApiKeyPlaceholder = updateApiKeyPlaceholder;
window.updateSettingsPlaceholder = updateSettingsPlaceholder;

// Start the app - check onboarding first
if (!hasCompletedOnboarding) {
  showOnboarding();
} else {
  init();
}
