import { GoogleGenerativeAI } from '@google/generative-ai';

let genAI = null;
let grokAvailable = false;
let aiProvider = null;

export function initializeAI() {
    if (aiProvider !== null) return aiProvider; // Already initialized

    // Try Gemini first
    if (process.env.GEMINI_API_KEY && process.env.GEMINI_API_KEY !== 'your_gemini_api_key_here') {
        genAI = new GoogleGenerativeAI(process.env.GEMINI_API_KEY);
        console.log('AI: Gemini API initialized');
        aiProvider = 'gemini';
        return 'gemini';
    }

    // Check Grok
    if (process.env.GROK_API_KEY && process.env.GROK_API_KEY !== 'your_grok_api_key_here') {
        grokAvailable = true;
        console.log('AI: Grok API available');
        aiProvider = 'grok';
        return 'grok';
    }

    console.log('AI: No API keys configured, using local analysis');
    aiProvider = 'local';
    return 'local';
}


/**
 * Query network data using natural language
 */
export async function queryWithAI(question, networkData) {
    // Prepare network context
    const networkSummary = createNetworkSummary(networkData);

    // Try Gemini
    if (genAI) {
        try {
            return await queryGemini(question, networkSummary, networkData);
        } catch (error) {
            console.error('Gemini error:', error.message);
        }
    }

    // Try Grok
    if (grokAvailable) {
        try {
            return await queryGrok(question, networkSummary, networkData);
        } catch (error) {
            console.error('Grok error:', error.message);
        }
    }

    // Fallback to local
    return localQuery(question, networkData);
}

/**
 * Query using Gemini
 */
async function queryGemini(question, networkSummary, networkData) {
    const model = genAI.getGenerativeModel({ model: 'gemini-2.0-flash' });


    const prompt = buildPrompt(question, networkSummary);
    const result = await model.generateContent(prompt);
    const response = result.response.text();

    return {
        answer: response,
        recommendations: extractRecommendations(response, networkData),
        source: 'gemini'
    };
}

/**
 * Query using Grok (X.AI)
 */
async function queryGrok(question, networkSummary, networkData) {
    const response = await fetch('https://api.x.ai/v1/chat/completions', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${process.env.GROK_API_KEY}`
        },
        body: JSON.stringify({
            model: 'grok-beta',
            messages: [
                {
                    role: 'system',
                    content: 'You are a professional networking advisor analyzing someone\'s social network data. Be concise and actionable.'
                },
                {
                    role: 'user',
                    content: buildPrompt(question, networkSummary)
                }
            ],
            max_tokens: 1000
        })
    });

    const data = await response.json();
    const answer = data.choices?.[0]?.message?.content || 'Unable to process query';

    return {
        answer,
        recommendations: extractRecommendations(answer, networkData),
        source: 'grok'
    };
}

/**
 * Build the AI prompt
 */
function buildPrompt(question, networkSummary) {
    return `You are a professional networking advisor analyzing someone's social network data.

Here is a summary of their network:
${networkSummary}

The user asks: "${question}"

Provide a helpful, actionable response with specific recommendations.
- Name specific people when relevant (use exact names from the data)
- Explain WHY each recommendation makes sense
- Suggest specific approaches for reaching out
- Focus on the most relevant 3-5 contacts
- Be concise but thorough`;
}

/**
 * Create a summary of network data for AI context
 */
function createNetworkSummary(networkData) {
    // Industry-tagged contacts for better context
    const taggedContacts = networkData.contacts
        .slice(0, 150)
        .map(c => {
            const industry = detectIndustry(c);
            const warmth = c.scores?.warmth || 'unknown';
            return `- ${c.name}: ${c.title || 'Unknown'} at ${c.company || 'Unknown'} [${c.platforms?.join(', ') || c.platform}] (Industry: ${industry}, Warmth: ${warmth})`;
        })
        .join('\n');

    const platformBreakdown = networkData.contacts.reduce((acc, c) => {
        const platforms = c.platforms || [c.platform];
        for (const p of platforms) {
            acc[p] = (acc[p] || 0) + 1;
        }
        return acc;
    }, {});

    // Find notable clusters
    const industries = networkData.contacts.reduce((acc, c) => {
        const ind = detectIndustry(c);
        acc[ind] = (acc[ind] || 0) + 1;
        return acc;
    }, {});

    return `Total contacts: ${networkData.contacts.length}
Platforms: ${Object.entries(platformBreakdown).map(([k, v]) => `${k}: ${v}`).join(', ')}
Industry breakdown: ${Object.entries(industries).sort((a, b) => b[1] - a[1]).slice(0, 5).map(([k, v]) => `${k}: ${v}`).join(', ')}

Contacts:
${taggedContacts}`;
}

/**
 * Detect industry from title/company
 */
function detectIndustry(contact) {
    const text = `${contact.title || ''} ${contact.company || ''}`.toLowerCase();

    const industries = {
        'Art': ['gallery', 'museum', 'curator', 'artist', 'art ', 'exhibition', 'collector', 'cultural', 'creative director'],
        'Gaming': ['game', 'gaming', 'esport', 'studio', 'developer', 'publisher', 'unity', 'unreal'],
        'Tech': ['software', 'engineer', 'developer', 'cto', 'tech', 'startup', 'saas', 'ai ', 'machine learning', 'data'],
        'Education': ['school', 'university', 'professor', 'teacher', 'education', 'academic', 'research'],
        'Finance': ['bank', 'finance', 'investor', 'vc', 'venture', 'capital', 'fund', 'trading'],
        'Media': ['journalist', 'writer', 'editor', 'media', 'news', 'content', 'podcast', 'influencer'],
        'Healthcare': ['doctor', 'medical', 'health', 'pharma', 'biotech', 'hospital'],
        'Consulting': ['consultant', 'consulting', 'advisor', 'strategy', 'mckkinsey', 'bcg', 'deloitte']
    };

    for (const [industry, keywords] of Object.entries(industries)) {
        if (keywords.some(kw => text.includes(kw))) {
            return industry;
        }
    }

    return 'Other';
}

/**
 * Local query without AI - pattern matching on keywords
 */
function localQuery(question, networkData) {
    const questionLower = question.toLowerCase();
    let recommendations = [];
    let answer = '';

    // Enhanced keyword detection with industry tags
    if (questionLower.includes('art') || questionLower.includes('gallery') || questionLower.includes('curator') || questionLower.includes('basel')) {
        recommendations = findByIndustry(networkData, 'Art');
        answer = `Based on your network, here are your strongest connections in the art world:`;
    } else if (questionLower.includes('game') || questionLower.includes('gaming') || questionLower.includes('local legends')) {
        recommendations = findByIndustry(networkData, 'Gaming');
        answer = `Based on your network, here are contacts in gaming/game development:`;
    } else if (questionLower.includes('fund') || questionLower.includes('investor') || questionLower.includes('vc') || questionLower.includes('money')) {
        recommendations = findByIndustry(networkData, 'Finance');
        answer = `Based on your network, here are potential funding/investor contacts:`;
    } else if (questionLower.includes('tech') || questionLower.includes('startup') || questionLower.includes('developer')) {
        recommendations = findByIndustry(networkData, 'Tech');
        answer = `Based on your network, here are tech industry contacts:`;
    } else if (questionLower.includes('education') || questionLower.includes('school') || questionLower.includes('conexkt')) {
        recommendations = findByIndustry(networkData, 'Education');
        answer = `Based on your network, here are education sector contacts:`;
    } else if (questionLower.includes('reconnect') || questionLower.includes('cold') || questionLower.includes('reach out')) {
        recommendations = findCoolingConnections(networkData);
        answer = `Here are valuable connections that are cooling down - consider reaching out:`;
    } else {
        // Generic search with industry detection
        const words = questionLower.split(/\s+/).filter(w => w.length > 3);
        recommendations = findByKeywords(networkData, words);
        answer = `Here are contacts that might be relevant to your query:`;
    }

    if (recommendations.length === 0) {
        answer = `I couldn't find specific matches for "${question}". Try searching for:
- Industries: "art", "gaming", "tech", "finance", "education"
- Actions: "reconnect", "reach out"
- Or specific company/role names`;
    }

    return {
        answer,
        recommendations,
        source: 'local'
    };
}

function findByIndustry(networkData, targetIndustry) {
    return networkData.contacts
        .filter(c => detectIndustry(c) === targetIndustry)
        .sort((a, b) => (b.scores?.warmth || 0) - (a.scores?.warmth || 0))
        .slice(0, 10)
        .map(c => ({
            name: c.name,
            company: c.company,
            title: c.title,
            industry: targetIndustry,
            warmth: c.scores?.warmth || 0,
            reason: `Works in ${targetIndustry} sector`
        }));
}

function findByKeywords(networkData, keywords) {
    return networkData.contacts
        .filter(c => {
            const text = `${c.name} ${c.company} ${c.title}`.toLowerCase();
            return keywords.some(kw => text.includes(kw));
        })
        .sort((a, b) => (b.scores?.warmth || 0) - (a.scores?.warmth || 0))
        .slice(0, 10)
        .map(c => ({
            name: c.name,
            company: c.company,
            title: c.title,
            industry: detectIndustry(c),
            warmth: c.scores?.warmth || 0,
            reason: `Matches keywords in profile`
        }));
}

function findCoolingConnections(networkData) {
    const now = Date.now();
    const sixtyDays = 60 * 24 * 60 * 60 * 1000;

    return networkData.contacts
        .filter(c => {
            const hasInteractions = c.interactions && c.interactions.length > 0;
            if (!hasInteractions) return false;

            const lastDate = c.interactions
                .map(i => i.lastDate || i.date)
                .filter(Boolean)
                .map(d => new Date(d).getTime())
                .sort((a, b) => b - a)[0];

            return lastDate && (now - lastDate) > sixtyDays;
        })
        .sort((a, b) => (b.scores?.vouch || 0) - (a.scores?.vouch || 0))
        .slice(0, 10)
        .map(c => ({
            name: c.name,
            company: c.company,
            title: c.title,
            industry: detectIndustry(c),
            warmth: c.scores?.warmth || 0,
            reason: `Had previous engagement, now cooling`
        }));
}

function extractRecommendations(response, networkData) {
    const recommendations = [];

    for (const contact of networkData.contacts.slice(0, 200)) {
        if (response.includes(contact.name)) {
            recommendations.push({
                name: contact.name,
                company: contact.company,
                title: contact.title,
                industry: detectIndustry(contact),
                warmth: contact.scores?.warmth || 0,
                reason: 'Mentioned in AI response'
            });
        }
    }

    return recommendations.slice(0, 5);
}

// Export for use in other modules
export { detectIndustry };
