import { scoreRelationship } from './relationship-score.js';
import { findWarmPaths } from './path-finder.js';

export const analyzeNetwork = {
    scoreRelationship,
    findWarmPaths,

    /**
     * Get overall network health summary
     */
    getHealthSummary(networkData) {
        const now = Date.now();
        const thirtyDays = 30 * 24 * 60 * 60 * 1000;
        const ninetyDays = 90 * 24 * 60 * 60 * 1000;

        let warm = 0;
        let cooling = 0;
        let cold = 0;

        for (const contact of networkData.contacts) {
            const lastInteraction = getLastInteractionDate(contact);
            if (!lastInteraction) {
                cold++;
                continue;
            }

            const daysSince = now - new Date(lastInteraction).getTime();

            if (daysSince < thirtyDays) {
                warm++;
            } else if (daysSince < ninetyDays) {
                cooling++;
            } else {
                cold++;
            }
        }

        return {
            warm,
            cooling,
            cold,
            total: networkData.contacts.length,
            warmPercentage: Math.round((warm / networkData.contacts.length) * 100) || 0
        };
    },

    /**
     * Get actionable alerts
     */
    getAlerts(networkData) {
        const alerts = [];
        const now = Date.now();
        const thirtyDays = 30 * 24 * 60 * 60 * 1000;
        const sixtyDays = 60 * 24 * 60 * 60 * 1000;
        const ninetyDays = 90 * 24 * 60 * 60 * 1000;

        for (const contact of networkData.contacts) {
            const scores = scoreRelationship(contact, networkData);
            const lastInteraction = getLastInteractionDate(contact);

            if (!lastInteraction) continue;

            const daysSince = now - new Date(lastInteraction).getTime();

            // Alert: High-value connection going cold
            if (scores.vouch >= 60 && daysSince > sixtyDays && daysSince < ninetyDays) {
                alerts.push({
                    type: 'cooling',
                    priority: 'high',
                    contact: contact.name,
                    message: `${contact.name} was a strong connection but you haven't connected in ${Math.round(daysSince / (24 * 60 * 60 * 1000))} days`,
                    suggestion: 'Consider reaching out with a quick check-in'
                });
            }

            // Alert: Someone who reached out first - opportunity to strengthen
            if (contact.connectionInitiator === 'them' && scores.warmth > 30 && daysSince > thirtyDays && daysSince < ninetyDays) {
                alerts.push({
                    type: 'opportunity',
                    priority: 'medium',
                    contact: contact.name,
                    message: `${contact.name} reached out to connect with you`,
                    suggestion: 'They showed interest first - nurture this relationship'
                });
            }

            // Alert: Recommendation received - high priority to reciprocate
            const recommendationsReceived = contact.interactions?.filter(i =>
                i.type === 'recommendation_received'
            ).length || 0;

            if (recommendationsReceived > 0) {
                alerts.push({
                    type: 'reciprocity',
                    priority: 'high',
                    contact: contact.name,
                    message: `${contact.name} wrote you a recommendation`,
                    suggestion: 'Consider writing them a recommendation back'
                });
            }

            // Alert: Endorsement reciprocity (existing)
            const endorsementsReceived = contact.interactions?.filter(i =>
                i.type === 'endorsement_received'
            ).length || 0;

            const endorsementsGiven = contact.interactions?.filter(i =>
                i.type === 'endorsement_given'
            ).length || 0;

            if (endorsementsReceived >= 3 && endorsementsGiven === 0) {
                alerts.push({
                    type: 'reciprocity',
                    priority: 'low',
                    contact: contact.name,
                    message: `${contact.name} has endorsed you ${endorsementsReceived} times`,
                    suggestion: 'Consider endorsing them back'
                });
            }

            // Alert: Unfinished conversation
            const hasUnfinished = contact.interactions?.some(i =>
                i.type === 'message' && i.hasDeepMessages &&
                new Date(i.lastDate).getTime() > now - ninetyDays
            );

            if (hasUnfinished && daysSince > sixtyDays) {
                alerts.push({
                    type: 'unfinished',
                    priority: 'medium',
                    contact: contact.name,
                    message: `You had a substantive conversation with ${contact.name}`,
                    suggestion: 'Check for natural re-engagement hooks'
                });
            }
        }

        // Sort by priority
        const priorityOrder = { high: 0, medium: 1, low: 2 };
        alerts.sort((a, b) => priorityOrder[a.priority] - priorityOrder[b.priority]);

        return alerts.slice(0, 20); // Top 20 alerts
    }
};


function getLastInteractionDate(contact) {
    if (!contact.interactions || contact.interactions.length === 0) {
        return contact.connectedDate || null;
    }

    const dates = contact.interactions
        .map(i => i.lastDate || i.date)
        .filter(Boolean)
        .map(d => new Date(d).getTime());

    if (dates.length === 0) {
        return contact.connectedDate || null;
    }

    return new Date(Math.max(...dates)).toISOString();
}
