import { scoreRelationship } from './relationship-score.js';

/**
 * Find warm paths to a target company, person, or goal
 */
export async function findWarmPaths(target, networkData) {
    const targetLower = target.toLowerCase();
    const paths = [];

    // Score all contacts
    const scoredContacts = networkData.contacts.map(contact => ({
        ...contact,
        scores: scoreRelationship(contact, networkData)
    }));

    // Find direct matches (work at target company, have target in title, etc.)
    const directMatches = scoredContacts.filter(c => {
        const company = (c.company || '').toLowerCase();
        const title = (c.title || '').toLowerCase();
        const name = (c.name || '').toLowerCase();

        return company.includes(targetLower) ||
            title.includes(targetLower) ||
            name.includes(targetLower);
    });

    // Score and rank direct matches
    for (const match of directMatches) {
        const relevance = calculateRelevance(match, targetLower);

        paths.push({
            type: 'direct',
            contact: {
                name: match.name,
                company: match.company,
                title: match.title,
                platforms: match.platforms
            },
            relevanceScore: relevance,
            warmthScore: match.scores.warmth,
            combinedScore: Math.round((relevance * 0.5 + match.scores.warmth * 0.5)),
            suggestedApproach: generateApproach(match, target)
        });
    }

    // Find industry/domain matches (for broader goals like "Art Basel")
    const industryKeywords = getIndustryKeywords(targetLower);
    if (industryKeywords.length > 0) {
        const industryMatches = scoredContacts.filter(c => {
            const company = (c.company || '').toLowerCase();
            const title = (c.title || '').toLowerCase();

            return industryKeywords.some(kw =>
                company.includes(kw) || title.includes(kw)
            );
        });

        for (const match of industryMatches) {
            // Skip if already in direct matches
            if (directMatches.some(d => d.name === match.name)) continue;

            const relevance = calculateIndustryRelevance(match, industryKeywords);

            paths.push({
                type: 'industry',
                contact: {
                    name: match.name,
                    company: match.company,
                    title: match.title,
                    platforms: match.platforms
                },
                relevanceScore: relevance,
                warmthScore: match.scores.warmth,
                combinedScore: Math.round((relevance * 0.4 + match.scores.warmth * 0.6)),
                suggestedApproach: generateApproach(match, target),
                matchReason: `Works in related field`
            });
        }
    }

    // Sort by combined score
    paths.sort((a, b) => b.combinedScore - a.combinedScore);

    return paths.slice(0, 15); // Top 15 paths
}

/**
 * Calculate relevance score for direct match
 */
function calculateRelevance(contact, target) {
    let score = 0;

    const company = (contact.company || '').toLowerCase();
    const title = (contact.title || '').toLowerCase();

    // Exact company match is best
    if (company === target) score += 100;
    else if (company.includes(target)) score += 70;

    // Title match (e.g., "curator")
    if (title.includes(target)) score += 60;

    // Seniority boost
    const seniorTerms = ['director', 'head', 'chief', 'vp', 'president', 'founder', 'ceo', 'partner'];
    if (seniorTerms.some(t => title.includes(t))) score += 20;

    return Math.min(100, score);
}

/**
 * Get industry keywords for broader goals
 */
function getIndustryKeywords(target) {
    const industryMap = {
        'art basel': ['gallery', 'curator', 'museum', 'art', 'contemporary', 'collector', 'exhibition'],
        'curator': ['gallery', 'museum', 'art', 'exhibition', 'cultural'],
        'funding': ['investor', 'vc', 'venture', 'capital', 'angel', 'fund'],
        'game': ['gaming', 'game', 'indie', 'studio', 'publisher', 'developer'],
        'local legends': ['gaming', 'tcg', 'card game', 'tabletop', 'crowdfunding'],
        'education': ['school', 'university', 'edtech', 'learning', 'teacher', 'academic']
    };

    for (const [key, keywords] of Object.entries(industryMap)) {
        if (target.includes(key)) {
            return keywords;
        }
    }

    return [];
}

/**
 * Calculate relevance for industry match
 */
function calculateIndustryRelevance(contact, keywords) {
    const company = (contact.company || '').toLowerCase();
    const title = (contact.title || '').toLowerCase();

    let matches = 0;
    for (const kw of keywords) {
        if (company.includes(kw)) matches++;
        if (title.includes(kw)) matches++;
    }

    return Math.min(100, matches * 25);
}

/**
 * Generate approach suggestion
 */
function generateApproach(contact, target) {
    const warmth = contact.scores?.warmth || 0;
    const vouch = contact.scores?.vouch || 0;

    if (warmth > 70) {
        return `Strong connection - reach out directly about ${target}`;
    } else if (warmth > 40) {
        return `Warm connection - re-engage first, then ask about ${target}`;
    } else if (vouch > 50) {
        return `They may remember you positively - remind them of your connection, then ask`;
    } else {
        return `Cold connection - reconnect with value before asking for help`;
    }
}
