/**
 * Relationship scoring based on half-life decay and interaction quality
 */

const HALF_LIFE_DAYS = 180; // Relationships lose 50% strength every 180 days
const MS_PER_DAY = 24 * 60 * 60 * 1000;

export function scoreRelationship(contact, networkData) {
    const warmth = calculateWarmth(contact);
    const vouch = calculateVouchScore(contact);
    const reciprocity = calculateReciprocity(contact);
    const crossPlatformBoost = calculateCrossPlatformBoost(contact);

    return {
        warmth: Math.round(warmth * (1 + crossPlatformBoost * 0.2)),
        vouch: Math.round(vouch),
        reciprocity: Math.round(reciprocity),
        overall: Math.round((warmth * 0.4 + vouch * 0.4 + Math.abs(reciprocity) * 0.2) * (1 + crossPlatformBoost * 0.1)),
        crossPlatformBoost: crossPlatformBoost > 0
    };
}

/**
 * Calculate warmth using half-life decay
 * Recent interactions = high warmth, old interactions = decayed warmth
 */
function calculateWarmth(contact) {
    const now = Date.now();
    const baseWarmth = 100;

    // Get most recent interaction date
    const lastInteraction = getMostRecentInteraction(contact);
    if (!lastInteraction) {
        // Use connection date with heavy decay
        if (contact.connectedDate) {
            const daysSince = (now - new Date(contact.connectedDate).getTime()) / MS_PER_DAY;
            return baseWarmth * Math.pow(0.5, daysSince / HALF_LIFE_DAYS);
        }
        return 10; // Minimal warmth for unknown connections
    }

    const daysSince = (now - new Date(lastInteraction).getTime()) / MS_PER_DAY;
    let warmth = baseWarmth * Math.pow(0.5, daysSince / HALF_LIFE_DAYS);

    // Boost for deep interactions
    const hasDeepInteractions = contact.interactions?.some(i => i.hasDeepMessages);
    if (hasDeepInteractions) {
        warmth *= 1.3;
    }

    // Boost for frequency
    const totalInteractions = contact.interactions?.reduce((sum, i) =>
        sum + (i.count || 1), 0
    ) || 0;

    if (totalInteractions > 10) warmth *= 1.2;
    if (totalInteractions > 50) warmth *= 1.1;

    return Math.min(100, warmth);
}

/**
 * Calculate vouch score - how likely they'd advocate for you
 */
function calculateVouchScore(contact) {
    let score = 0;

    // Recommendations are huge
    const recommendations = contact.interactions?.filter(i =>
        i.type === 'recommendation_received'
    ).length || 0;
    score += recommendations * 30;

    // Endorsements matter
    const endorsements = contact.interactions?.filter(i =>
        i.type === 'endorsement_received'
    ).length || 0;
    score += endorsements * 5;

    // Deep conversations indicate advocacy potential
    const deepConversations = contact.interactions?.filter(i =>
        i.hasDeepMessages
    ).length || 0;
    score += deepConversations * 10;

    // Message volume
    const messageVolume = contact.interactions?.reduce((sum, i) =>
        i.type === 'message' ? sum + (i.count || 0) : sum, 0
    ) || 0;
    score += Math.min(20, messageVolume / 5);

    // Recency boost
    const lastInteraction = getMostRecentInteraction(contact);
    if (lastInteraction) {
        const daysSince = (Date.now() - new Date(lastInteraction).getTime()) / MS_PER_DAY;
        if (daysSince < 30) score *= 1.3;
        else if (daysSince < 90) score *= 1.1;
        else if (daysSince > 365) score *= 0.7;
    }

    return Math.min(100, score);
}

/**
 * Calculate reciprocity balance
 * Positive = they owe you, Negative = you owe them
 */
function calculateReciprocity(contact) {
    let balance = 0;

    // Recommendations received from them: you owe them
    const recsReceived = contact.interactions?.filter(i =>
        i.type === 'recommendation_received'
    ).length || 0;
    balance -= recsReceived * 10;

    // Endorsements received from them: you owe them
    const endorsementsReceived = contact.interactions?.filter(i =>
        i.type === 'endorsement_received'
    ).length || 0;
    balance -= endorsementsReceived * 2;

    // Endorsements you gave them: they owe you
    const endorsementsGiven = contact.interactions?.filter(i =>
        i.type === 'endorsement_given'
    ).length || 0;
    balance += endorsementsGiven * 2;

    // Connection initiator matters
    if (contact.connectionInitiator === 'them') {
        balance += 5; // They reached out first
    } else if (contact.connectionInitiator === 'you') {
        balance -= 3; // You reached out first
    }

    return balance;
}


/**
 * Cross-platform boost - same person on multiple platforms is stronger
 */
function calculateCrossPlatformBoost(contact) {
    const platformCount = contact.platforms?.length || 1;
    return platformCount > 1 ? (platformCount - 1) * 0.5 : 0;
}

function getMostRecentInteraction(contact) {
    if (!contact.interactions || contact.interactions.length === 0) {
        return null;
    }

    const dates = contact.interactions
        .map(i => i.lastDate || i.date)
        .filter(Boolean)
        .map(d => new Date(d).getTime());

    if (dates.length === 0) return null;

    return new Date(Math.max(...dates)).toISOString();
}
