/**
 * Bluesky API utilities for resolving DIDs to handles
 */

const BSKY_API = 'https://public.api.bsky.app';

/**
 * Resolve a DID to a handle and display name
 */
export async function resolveBlueskyDID(did) {
    try {
        const response = await fetch(`${BSKY_API}/xrpc/app.bsky.actor.getProfile?actor=${encodeURIComponent(did)}`);
        if (!response.ok) return null;

        const data = await response.json();
        return {
            did: data.did,
            handle: data.handle,
            displayName: data.displayName || data.handle,
            avatar: data.avatar,
            description: data.description,
            followersCount: data.followersCount,
            followsCount: data.followsCount
        };
    } catch (error) {
        console.error(`Failed to resolve DID ${did}:`, error.message);
        return null;
    }
}

/**
 * Batch resolve multiple DIDs (with rate limiting)
 */
export async function batchResolveBlueskyDIDs(dids, onProgress) {
    const results = new Map();
    const batchSize = 5;
    const delay = 200; // ms between batches to avoid rate limiting

    for (let i = 0; i < dids.length; i += batchSize) {
        const batch = dids.slice(i, i + batchSize);

        const promises = batch.map(did => resolveBlueskyDID(did));
        const batchResults = await Promise.all(promises);

        batch.forEach((did, idx) => {
            if (batchResults[idx]) {
                results.set(did, batchResults[idx]);
            }
        });

        if (onProgress) {
            onProgress(Math.min(i + batchSize, dids.length), dids.length);
        }

        // Rate limit delay
        if (i + batchSize < dids.length) {
            await new Promise(resolve => setTimeout(resolve, delay));
        }
    }

    return results;
}
