import express from 'express';
import cors from 'cors';
import { fileURLToPath } from 'url';
import { dirname, join } from 'path';
import dotenv from 'dotenv';
import fs from 'fs';
import multer from 'multer';
import { loadAllData } from './parsers/index.js';
import { analyzeNetwork } from './analysis/index.js';
import { queryWithAI, detectIndustry, initializeAI } from './ai/gemini.js';

dotenv.config();

// Initialize AI after dotenv loads
initializeAI();


const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

const app = express();
const PORT = process.env.PORT || 3001;

app.use(cors());
app.use(express.json());

// Configure multer for file uploads
const dataDir = join(__dirname, '..', 'data');

// Ensure data directory exists
if (!fs.existsSync(dataDir)) {
    fs.mkdirSync(dataDir, { recursive: true });
}

const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, dataDir);
    },
    filename: (req, file, cb) => {
        // Keep original filename but add timestamp to avoid collisions
        const timestamp = Date.now();
        cb(null, `${timestamp}-${file.originalname}`);
    }
});

const upload = multer({
    storage,
    limits: { fileSize: 100 * 1024 * 1024 }, // 100MB limit
    fileFilter: (req, file, cb) => {
        // Accept zip files, json, csv
        const allowedTypes = ['.zip', '.json', '.csv', '.car'];
        const ext = file.originalname.toLowerCase().slice(file.originalname.lastIndexOf('.'));
        if (allowedTypes.includes(ext)) {
            cb(null, true);
        } else {
            cb(new Error(`File type ${ext} not allowed. Accepted: ${allowedTypes.join(', ')}`));
        }
    }
});

// Store loaded network data in memory
let networkData = null;

// Load data on startup
async function initializeData() {
    try {
        console.log('Loading network data...');
        networkData = await loadAllData(join(__dirname, '..', 'data'));
        console.log(`Loaded ${networkData.contacts.length} contacts from ${networkData.platforms.length} platforms`);
    } catch (error) {
        console.error('Error loading data:', error.message);
        networkData = { contacts: [], platforms: [], messages: [], relationships: [] };
    }
}

// API Routes

// File upload endpoint
app.post('/api/upload', upload.array('files', 10), async (req, res) => {
    try {
        if (!req.files || req.files.length === 0) {
            return res.status(400).json({ error: 'No files uploaded' });
        }

        const uploadedFiles = req.files.map(f => ({
            filename: f.filename,
            originalName: f.originalname,
            size: f.size,
            path: f.path
        }));

        console.log(`Uploaded ${uploadedFiles.length} file(s):`, uploadedFiles.map(f => f.originalName));

        // Process each uploaded file
        for (const file of uploadedFiles) {
            await processUploadedFile(file.path, file.originalName);
        }

        // Reload data after processing
        await initializeData();

        res.json({
            success: true,
            message: `Processed ${uploadedFiles.length} file(s). Found ${networkData?.contacts?.length || 0} contacts.`,
            files: uploadedFiles,
            contactsLoaded: networkData?.contacts?.length || 0
        });
    } catch (error) {
        console.error('Upload error:', error);
        res.status(500).json({ error: error.message || 'Upload failed' });
    }
});

// Process an uploaded file - extract and move to correct platform folder
async function processUploadedFile(filePath, originalName) {
    const { exec } = await import('child_process');
    const { promisify } = await import('util');
    const execAsync = promisify(exec);

    const lowerName = originalName.toLowerCase();

    // If it's a ZIP file, extract it
    if (lowerName.endsWith('.zip')) {
        console.log(`Extracting ${originalName}...`);

        // Create temp extraction folder
        const tempDir = filePath.replace('.zip', '').replace('.ZIP', '') + '_temp';

        try {
            // Use ditto on Mac for proper Unicode handling, unzip on Linux
            const isMac = process.platform === 'darwin';
            if (isMac) {
                // ditto handles Unicode filenames properly on Mac
                await execAsync(`mkdir -p "${tempDir}" && ditto -x -k "${filePath}" "${tempDir}"`);
            } else {
                // On Linux, use unzip with UTF-8 locale
                await execAsync(`mkdir -p "${tempDir}" && LANG=en_US.UTF-8 unzip -o -q "${filePath}" -d "${tempDir}"`, {
                    env: { ...process.env, LANG: 'en_US.UTF-8' }
                });
            }
            console.log(`Extracted to temp: ${tempDir}`);

            // Check for nested ZIPs first (LinkedIn double-zip issue)
            const tempFiles = fs.readdirSync(tempDir);
            for (const f of tempFiles) {
                const nestedPath = join(tempDir, f);
                if (f.toLowerCase().endsWith('.zip') && fs.statSync(nestedPath).isFile()) {
                    console.log(`Found nested ZIP: ${f}, extracting...`);
                    if (isMac) {
                        await execAsync(`ditto -x -k "${nestedPath}" "${tempDir}"`);
                    } else {
                        await execAsync(`LANG=en_US.UTF-8 unzip -o -q "${nestedPath}" -d "${tempDir}"`, {
                            env: { ...process.env, LANG: 'en_US.UTF-8' }
                        });
                    }
                }
            }

            // Now detect platform and move files
            const platform = detectPlatform(tempDir, originalName);
            if (platform) {
                const targetDir = join(dataDir, platform);
                await execAsync(`mkdir -p "${targetDir}"`);

                // Move all files to platform folder
                await execAsync(`cp -R "${tempDir}"/* "${targetDir}/" 2>/dev/null || true`);

                // Also check subdirectories (LinkedIn puts files in subfolders sometimes)
                const subdirs = fs.readdirSync(tempDir).filter(f =>
                    fs.statSync(join(tempDir, f)).isDirectory()
                );
                for (const subdir of subdirs) {
                    await execAsync(`cp -R "${tempDir}/${subdir}"/* "${targetDir}/" 2>/dev/null || true`);
                }

                console.log(`Moved ${platform} data to: ${targetDir}`);
            }

            // Cleanup temp folder
            await execAsync(`rm -rf "${tempDir}"`);

        } catch (error) {
            console.error(`Failed to process ${originalName}:`, error.message);
        }
    } else if (lowerName.endsWith('.csv') || lowerName.endsWith('.json')) {
        // For individual CSV/JSON files, detect platform and copy directly
        const platform = detectPlatformFromFilename(originalName);
        if (platform) {
            const targetDir = join(dataDir, platform);
            if (!fs.existsSync(targetDir)) {
                fs.mkdirSync(targetDir, { recursive: true });
            }
            fs.copyFileSync(filePath, join(targetDir, originalName));
            console.log(`Copied ${originalName} to ${platform} folder`);
        }
    }
}

// Detect platform from extracted folder contents or filename
function detectPlatform(extractedDir, originalName) {
    const lowerName = originalName.toLowerCase();

    // Check filename hints first
    if (lowerName.includes('linkedin')) return 'linkedin';
    if (lowerName.includes('facebook') || lowerName.includes('fb')) return 'facebook';
    if (lowerName.includes('instagram')) return 'instagram';
    if (lowerName.includes('twitter') || lowerName.includes('-x-')) return 'twitter';
    if (lowerName.includes('bluesky') || lowerName.includes('bsky')) return 'bluesky';
    if (lowerName.includes('threads')) return 'threads';
    if (lowerName.includes('google') || lowerName.includes('contacts')) return 'google';

    // Check folder contents for platform-specific files
    try {
        const files = getAllFilesRecursive(extractedDir);

        // LinkedIn indicators
        if (files.some(f => f.toLowerCase().includes('connections.csv'))) return 'linkedin';
        if (files.some(f => f.toLowerCase().includes('endorsement'))) return 'linkedin';

        // Facebook indicators
        if (files.some(f => f.toLowerCase().includes('friends.json'))) return 'facebook';
        if (files.some(f => f.toLowerCase().includes('your_friends'))) return 'facebook';

        // Instagram indicators
        if (files.some(f => f.toLowerCase().includes('followers_and_following'))) return 'instagram';

        // Twitter indicators
        if (files.some(f => f.toLowerCase().includes('follower.js'))) return 'twitter';
        if (files.some(f => f.toLowerCase().includes('following.js'))) return 'twitter';

    } catch (e) {
        console.error('Error detecting platform:', e.message);
    }

    // Default to linkedin if we can't detect (most common)
    console.log('Could not detect platform, defaulting to linkedin');
    return 'linkedin';
}

function detectPlatformFromFilename(filename) {
    const lower = filename.toLowerCase();
    if (lower.includes('linkedin') || lower.includes('connection')) return 'linkedin';
    if (lower.includes('facebook') || lower.includes('friend')) return 'facebook';
    if (lower.includes('instagram') || lower.includes('follower')) return 'instagram';
    if (lower.includes('twitter')) return 'twitter';
    return 'linkedin'; // default
}

function getAllFilesRecursive(dir) {
    let files = [];
    try {
        const items = fs.readdirSync(dir);
        for (const item of items) {
            const fullPath = join(dir, item);
            if (fs.statSync(fullPath).isDirectory()) {
                files = files.concat(getAllFilesRecursive(fullPath));
            } else {
                files.push(item);
            }
        }
    } catch (e) {
        // Ignore errors
    }
    return files;
}

// Reload data endpoint (for re-parsing after manual file changes)
app.post('/api/reload', async (req, res) => {
    try {
        await initializeData();
        res.json({
            success: true,
            contactsLoaded: networkData?.contacts?.length || 0,
            platforms: networkData?.platforms || []
        });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// Get network overview
app.get('/api/overview', (req, res) => {
    if (!networkData) {
        return res.json({ status: 'loading', contacts: 0, platforms: [] });
    }

    const overview = {
        totalContacts: networkData.contacts.length,
        platforms: networkData.platforms,
        platformBreakdown: networkData.contacts.reduce((acc, c) => {
            acc[c.platform] = (acc[c.platform] || 0) + 1;
            return acc;
        }, {}),
        relationshipHealth: analyzeNetwork.getHealthSummary(networkData),
        lastUpdated: new Date().toISOString()
    };

    res.json(overview);
});

// Get all contacts with scores
app.get('/api/contacts', (req, res) => {
    if (!networkData) {
        return res.json([]);
    }

    const scoredContacts = networkData.contacts.map(contact => ({
        ...contact,
        scores: analyzeNetwork.scoreRelationship(contact, networkData)
    }));

    // Sort by warmth score
    scoredContacts.sort((a, b) => b.scores.warmth - a.scores.warmth);

    res.json(scoredContacts);
});

// Get relationship alerts (cooling down, need attention)
app.get('/api/alerts', (req, res) => {
    if (!networkData) {
        return res.json([]);
    }

    const alerts = analyzeNetwork.getAlerts(networkData);
    res.json(alerts);
});

// Natural language query endpoint
app.post('/api/query', async (req, res) => {
    const { question } = req.body;

    if (!question) {
        return res.status(400).json({ error: 'Question is required' });
    }

    if (!networkData || networkData.contacts.length === 0) {
        return res.json({
            answer: 'No network data loaded yet. Please add your data exports to the /data folder.',
            recommendations: []
        });
    }

    try {
        const result = await queryWithAI(question, networkData);
        res.json(result);
    } catch (error) {
        console.error('AI query error:', error);
        res.status(500).json({ error: 'Failed to process query' });
    }
});

// Warm path finder
app.post('/api/warm-path', async (req, res) => {
    const { target } = req.body;

    if (!target) {
        return res.status(400).json({ error: 'Target company or person is required' });
    }

    if (!networkData) {
        return res.json({ paths: [] });
    }

    try {
        const paths = await analyzeNetwork.findWarmPaths(target, networkData);
        res.json({ paths });
    } catch (error) {
        console.error('Path finding error:', error);
        res.status(500).json({ error: 'Failed to find paths' });
    }
});

// Get industry breakdown
app.get('/api/industries', (req, res) => {
    if (!networkData) {
        return res.json({});
    }



    const industries = networkData.contacts.reduce((acc, c) => {
        const industry = detectIndustry ? detectIndustry(c) : 'Other';
        if (!acc[industry]) {
            acc[industry] = { count: 0, contacts: [] };
        }
        acc[industry].count++;
        if (acc[industry].contacts.length < 5) {
            acc[industry].contacts.push({
                name: c.name,
                title: c.title,
                company: c.company
            });
        }
        return acc;
    }, {});

    res.json(industries);
});

// Weekly digest - suggested reach-outs
app.get('/api/digest', (req, res) => {
    if (!networkData) {
        return res.json({ suggestions: [] });
    }

    const now = Date.now();
    const suggestions = [];

    // Find high-value cooling connections
    const coolingHighValue = networkData.contacts
        .map(c => ({
            ...c,
            scores: analyzeNetwork.scoreRelationship(c, networkData)
        }))
        .filter(c => {
            const warmth = c.scores.warmth;
            const vouch = c.scores.vouch;
            return warmth > 20 && warmth < 60 && vouch > 30;
        })
        .sort((a, b) => b.scores.vouch - a.scores.vouch)
        .slice(0, 5);

    for (const contact of coolingHighValue) {
        suggestions.push({
            type: 'reconnect',
            priority: 'high',
            contact: {
                name: contact.name,
                title: contact.title,
                company: contact.company
            },
            reason: 'High-value connection cooling down',
            action: `Reach out to ${contact.name.split(' ')[0]} with a quick check-in`
        });
    }

    // Find reciprocity opportunities
    const reciprocityOpps = networkData.contacts
        .map(c => ({
            ...c,
            scores: analyzeNetwork.scoreRelationship(c, networkData)
        }))
        .filter(c => c.scores.reciprocity < -5)
        .sort((a, b) => a.scores.reciprocity - b.scores.reciprocity)
        .slice(0, 3);

    for (const contact of reciprocityOpps) {
        suggestions.push({
            type: 'reciprocate',
            priority: 'medium',
            contact: {
                name: contact.name,
                title: contact.title,
                company: contact.company
            },
            reason: 'They\'ve supported you - time to reciprocate',
            action: `Consider endorsing or recommending ${contact.name.split(' ')[0]}`
        });
    }

    res.json({
        generatedAt: new Date().toISOString(),
        weekOf: new Date().toLocaleDateString('en-US', { month: 'long', day: 'numeric', year: 'numeric' }),
        suggestions
    });
});

// Network graph data for visualization
app.get('/api/graph', (req, res) => {
    if (!networkData) {
        return res.json({ nodes: [], edges: [] });
    }



    // Create nodes from top contacts
    const topContacts = networkData.contacts
        .map(c => ({
            ...c,
            scores: analyzeNetwork.scoreRelationship(c, networkData)
        }))
        .sort((a, b) => b.scores.overall - a.scores.overall)
        .slice(0, 50);

    const nodes = topContacts.map((c, i) => ({
        id: c.id || i,
        name: c.name,
        company: c.company,
        title: c.title,
        industry: detectIndustry ? detectIndustry(c) : 'Other',
        warmth: c.scores.warmth,
        vouch: c.scores.vouch,
        size: Math.max(10, c.scores.overall / 2)
    }));

    // Create edges based on shared companies
    const edges = [];
    const companyMap = new Map();

    topContacts.forEach((c, i) => {
        if (c.company) {
            if (!companyMap.has(c.company)) {
                companyMap.set(c.company, []);
            }
            companyMap.get(c.company).push(i);
        }
    });

    for (const [company, indices] of companyMap) {
        if (indices.length > 1) {
            for (let i = 0; i < indices.length - 1; i++) {
                edges.push({
                    source: nodes[indices[i]].id,
                    target: nodes[indices[i + 1]].id,
                    type: 'company',
                    label: company
                });
            }
        }
    }

    res.json({ nodes, edges });
});

// Company Intelligence - who works at companies you follow
app.get('/api/company-intel', (req, res) => {
    if (!networkData || !networkData.followedCompanies) {
        return res.json({ companies: [], contacts: [] });
    }

    const result = networkData.followedCompanies.map(company => {
        const companyName = company.name.toLowerCase();

        // Find contacts at this company
        const contactsAtCompany = networkData.contacts
            .filter(c => c.company?.toLowerCase().includes(companyName))
            .map(c => ({
                name: c.name,
                title: c.title,
                company: c.company,
                warmth: analyzeNetwork.scoreRelationship(c, networkData).warmth
            }))
            .sort((a, b) => b.warmth - a.warmth)
            .slice(0, 5);

        return {
            company: company.name,
            followedOn: company.followedOn,
            contactCount: contactsAtCompany.length,
            contacts: contactsAtCompany
        };
    }).filter(c => c.contactCount > 0);

    res.json({
        totalFollowed: networkData.followedCompanies.length,
        withConnections: result.length,
        companies: result
    });
});

// Smart prompts - personalized based on user's data
app.get('/api/smart-prompts', (req, res) => {
    if (!networkData || networkData.contacts.length === 0) {
        return res.json({ prompts: [], hasData: false });
    }

    const prompts = [];

    // Analyze top companies in their network
    const companyCounts = {};
    networkData.contacts.forEach(c => {
        if (c.company) {
            companyCounts[c.company] = (companyCounts[c.company] || 0) + 1;
        }
    });
    const topCompanies = Object.entries(companyCounts)
        .sort((a, b) => b[1] - a[1])
        .slice(0, 3)
        .map(([name]) => name);

    // Analyze top industries
    const industryCounts = {};
    networkData.contacts.forEach(c => {
        const industry = detectIndustry ? detectIndustry(c) : 'Other';
        if (industry && industry !== 'Other') {
            industryCounts[industry] = (industryCounts[industry] || 0) + 1;
        }
    });
    const topIndustries = Object.entries(industryCounts)
        .sort((a, b) => b[1] - a[1])
        .slice(0, 2)
        .map(([name]) => name);

    // Check for followed companies
    const followedCompanies = networkData.followedCompanies?.slice(0, 3) || [];

    // Generate personalized prompts
    prompts.push({
        text: 'Who should I reconnect with?',
        type: 'reconnect'
    });

    if (topCompanies[0]) {
        prompts.push({
            text: `Who at ${topCompanies[0]} should I know?`,
            type: 'company'
        });
    }

    if (topIndustries[0]) {
        prompts.push({
            text: `Who in ${topIndustries[0]} can help me?`,
            type: 'industry'
        });
    }

    if (followedCompanies[0]) {
        prompts.push({
            text: `Who knows people at ${followedCompanies[0].name}?`,
            type: 'path'
        });
    }

    prompts.push({
        text: 'Who owes me a response?',
        type: 'reciprocity'
    });

    // Add "Who knows..." expertise prompts
    const expertisePrompts = [
        'Who knows a good lawyer?',
        'Who knows investors or VCs?',
        'Who knows a designer?'
    ];
    const randomExpertise = expertisePrompts[Math.floor(Math.random() * expertisePrompts.length)];
    prompts.push({
        text: randomExpertise,
        type: 'expertise'
    });

    // Add some variety based on network size
    if (networkData.contacts.length > 1000) {
        prompts.push({
            text: 'Who are my most valuable connections?',
            type: 'analysis'
        });
    }

    res.json({
        prompts: prompts.slice(0, 4),
        hasData: true,
        stats: {
            totalContacts: networkData.contacts.length,
            topIndustry: topIndustries[0] || null,
            topCompany: topCompanies[0] || null
        }
    });
});

// Goal-specific query mode
app.post('/api/query-goal', async (req, res) => {
    const { question, mode } = req.body;

    if (!question) {
        return res.status(400).json({ error: 'Question required' });
    }

    if (!networkData) {
        return res.json({ answer: 'Data still loading...', recommendations: [] });
    }

    // Enhance question based on mode
    let enhancedQuestion = question;
    let industryFilter = null;

    if (mode === 'art') {
        // Avery Lake context
        enhancedQuestion = `[Context: Avery Lake is a Swiss-Canadian artist exploring humanity in the AI age, seeking gallery connections, curator relationships, and art world networking for shows like Art Basel]
        
Question: ${question}

Focus on: galleries, curators, artists, museums, collectors, art publications, cultural institutions`;
        industryFilter = ['Art', 'Media'];
    } else if (mode === 'business') {
        // Conexkt context
        enhancedQuestion = `[Context: Conexkt is a B Corporation consulting firm specializing in strategy, AI implementation, digital transformation, and science communication. Based in Canada and Switzerland.]
        
Question: ${question}

Focus on: education tech, innovation labs, schools, universities, tech companies, consulting opportunities`;
        industryFilter = ['Education', 'Tech', 'Consulting'];
    }

    try {
        const result = await queryWithAI(enhancedQuestion, networkData);

        // Filter recommendations by industry if mode specified
        if (industryFilter && result.recommendations) {
            result.recommendations = result.recommendations.filter(r =>
                industryFilter.includes(r.industry) || !r.industry
            );
        }

        res.json(result);
    } catch (error) {
        console.error('Goal query error:', error);
        res.status(500).json({ error: 'Query failed' });
    }
});

// "Who knows..." - Expertise discovery endpoint
app.post('/api/who-knows', async (req, res) => {
    const { expertise } = req.body;

    if (!expertise) {
        return res.status(400).json({ error: 'Expertise query required (e.g., "a lawyer", "someone at Google", "VCs")' });
    }

    if (!networkData || networkData.contacts.length === 0) {
        return res.json({
            matches: [],
            message: 'No data loaded yet. Import your network data first.'
        });
    }

    // Expertise keywords mapping
    const expertiseMap = {
        // Legal
        'lawyer': ['lawyer', 'attorney', 'legal', 'counsel', 'paralegal', 'law firm', 'juris', 'barrister', 'solicitor'],
        'legal': ['lawyer', 'attorney', 'legal', 'counsel', 'paralegal', 'law firm', 'juris', 'barrister', 'solicitor'],

        // Finance
        'vc': ['venture', 'capital', 'investor', 'investment', 'vc', 'partner at', 'fund', 'angels'],
        'investor': ['venture', 'capital', 'investor', 'investment', 'vc', 'partner at', 'fund', 'angels', 'portfolio'],
        'finance': ['finance', 'financial', 'cfo', 'accountant', 'banking', 'investment', 'wealth'],
        'accountant': ['accountant', 'cpa', 'accounting', 'bookkeeper', 'tax'],

        // Tech
        'developer': ['developer', 'engineer', 'programmer', 'software', 'coding', 'frontend', 'backend', 'fullstack'],
        'engineer': ['engineer', 'engineering', 'developer', 'software', 'technical'],
        'designer': ['designer', 'design', 'ux', 'ui', 'creative', 'graphic', 'product design'],
        'product': ['product manager', 'product lead', 'pm', 'product owner'],

        // Sales & Marketing
        'sales': ['sales', 'business development', 'account executive', 'bd', 'revenue'],
        'marketing': ['marketing', 'growth', 'brand', 'content', 'digital marketing', 'cmx'],
        'pr': ['public relations', 'pr', 'communications', 'press', 'media relations'],

        // Creative
        'artist': ['artist', 'art', 'gallery', 'curator', 'painter', 'sculptor', 'creative'],
        'curator': ['curator', 'gallery', 'museum', 'art director', 'exhibition'],
        'writer': ['writer', 'author', 'journalist', 'editor', 'content', 'copywriter'],

        // Healthcare
        'doctor': ['doctor', 'physician', 'md', 'medical', 'surgeon', 'healthcare'],
        'therapist': ['therapist', 'psychologist', 'counselor', 'mental health'],

        // Education
        'professor': ['professor', 'academic', 'researcher', 'university', 'faculty', 'phd'],
        'teacher': ['teacher', 'educator', 'instructor', 'teaching'],

        // Leadership
        'ceo': ['ceo', 'chief executive', 'founder', 'co-founder', 'president'],
        'founder': ['founder', 'co-founder', 'entrepreneur', 'started', 'ceo'],
        'executive': ['ceo', 'cfo', 'cto', 'coo', 'chief', 'vp', 'vice president', 'director', 'head of'],

        // Other
        'recruiter': ['recruiter', 'talent', 'hiring', 'hr', 'people ops', 'human resources'],
        'consultant': ['consultant', 'consulting', 'advisor', 'strategy'],
        'coach': ['coach', 'coaching', 'mentor', 'trainer']
    };

    // Normalize query
    const queryLower = expertise.toLowerCase().replace(/^(a |an |the |someone who is |someone in |people in )/, '');

    // Find matching keywords
    let searchTerms = [];
    for (const [key, terms] of Object.entries(expertiseMap)) {
        if (queryLower.includes(key)) {
            searchTerms = [...searchTerms, ...terms];
        }
    }

    // If no mapping found, use the query itself as search term
    if (searchTerms.length === 0) {
        searchTerms = queryLower.split(/\s+/).filter(t => t.length > 2);
    }

    // Search contacts
    const matches = networkData.contacts
        .map(contact => {
            let score = 0;
            const matchedOn = [];

            const titleLower = (contact.title || '').toLowerCase();
            const companyLower = (contact.company || '').toLowerCase();
            const industry = detectIndustry ? detectIndustry(contact) : 'Other';

            // Check job title
            for (const term of searchTerms) {
                if (titleLower.includes(term)) {
                    score += 10;
                    matchedOn.push(`Title contains "${term}"`);
                }
            }

            // Check company
            for (const term of searchTerms) {
                if (companyLower.includes(term)) {
                    score += 5;
                    matchedOn.push(`Works at company matching "${term}"`);
                }
            }

            // Industry match (lighter weight)
            if (industry && industry.toLowerCase().includes(queryLower)) {
                score += 3;
                matchedOn.push(`Industry: ${industry}`);
            }

            // Get warmth score
            const warmth = analyzeNetwork.scoreRelationship(contact, networkData).warmth;

            return {
                name: contact.name,
                title: contact.title,
                company: contact.company,
                industry,
                platform: contact.platform,
                profileUrl: contact.profileUrl,
                matchScore: score,
                warmth,
                matchedOn,
                // Combined score (relevance + relationship strength)
                combinedScore: score * 2 + warmth
            };
        })
        .filter(c => c.matchScore > 0)
        .sort((a, b) => b.combinedScore - a.combinedScore)
        .slice(0, 10);

    // Generate AI summary if we have matches
    let aiSummary = null;
    if (matches.length > 0) {
        try {
            const topNames = matches.slice(0, 5).map(m => `${m.name} (${m.title} at ${m.company})`).join(', ');
            aiSummary = `Found ${matches.length} people who might know about ${expertise}. Your warmest connections are: ${topNames}`;
        } catch (e) {
            // Skip AI summary if it fails
        }
    }

    res.json({
        query: expertise,
        matchCount: matches.length,
        matches,
        summary: aiSummary || (matches.length > 0
            ? `Found ${matches.length} matches for "${expertise}"`
            : `No direct matches for "${expertise}". Try a broader search or ask the AI directly.`),
        suggestedFollowUp: matches.length === 0
            ? `Try asking: "Who works in ${expertise}?" or check the full network with the AI query.`
            : null
    });
});

// Start server
app.listen(PORT, async () => {
    console.log(`Network Intelligence server running on http://localhost:${PORT}`);
    await initializeData();
});
