import { readFileSync, existsSync, createReadStream } from 'fs';
import { join } from 'path';
import { CarReader } from '@ipld/car';
import * as dagCbor from '@ipld/dag-cbor';

/**
 * Parse Bluesky data export
 * Bluesky exports data as CAR files (Content Addressable aRchive)
 */
export async function parseBluesky(dataDir) {
    const contacts = [];
    const messages = [];

    // Try to parse CAR file (repo.car)
    const carFile = join(dataDir, 'repo.car');
    if (existsSync(carFile)) {
        try {
            console.log('  - Parsing Bluesky CAR file...');
            const carData = readFileSync(carFile);
            const reader = await CarReader.fromBytes(carData);

            // Extract follows from CAR blocks
            for await (const block of reader.blocks()) {
                try {
                    const decoded = dagCbor.decode(block.bytes);

                    // Look for follow records (app.bsky.graph.follow)
                    if (decoded && decoded.$type === 'app.bsky.graph.follow') {
                        const did = decoded.subject;
                        contacts.push({
                            name: did, // DID as placeholder, would need API to resolve handle
                            did: did,
                            handle: null,
                            platform: 'bluesky',
                            connectedDate: decoded.createdAt || null,
                            interactions: []
                        });
                    }

                    // Look for profile records
                    if (decoded && decoded.$type === 'app.bsky.actor.profile') {
                        // This is the user's own profile
                        console.log('  - Found profile:', decoded.displayName || 'No name');
                    }
                } catch (e) {
                    // Not all blocks are CBOR decodable, skip silently
                }
            }

            console.log(`  - Found ${contacts.length} Bluesky follows`);
        } catch (e) {
            console.error('Error parsing Bluesky CAR:', e.message);
        }
    }

    // Also try JSON fallback (from API export)
    const followsFile = join(dataDir, 'follows.json');
    if (existsSync(followsFile)) {
        try {
            const data = JSON.parse(readFileSync(followsFile, 'utf-8'));
            const follows = Array.isArray(data) ? data : data.follows || [];

            for (const follow of follows) {
                // Avoid duplicates
                if (!contacts.some(c => c.did === follow.did)) {
                    contacts.push({
                        name: follow.displayName || follow.handle || 'Unknown',
                        handle: follow.handle,
                        did: follow.did,
                        platform: 'bluesky',
                        interactions: []
                    });
                }
            }
        } catch (e) {
            console.error('Error parsing Bluesky follows.json:', e.message);
        }
    }

    // Try to find followers.json
    const followersFile = join(dataDir, 'followers.json');
    if (existsSync(followersFile)) {
        try {
            const data = JSON.parse(readFileSync(followersFile, 'utf-8'));
            const followers = Array.isArray(data) ? data : data.followers || [];

            for (const follower of followers) {
                const existing = contacts.find(c => c.handle === follower.handle || c.did === follower.did);
                if (existing) {
                    existing.isMutual = true;
                } else {
                    contacts.push({
                        name: follower.displayName || follower.handle || 'Unknown',
                        handle: follower.handle,
                        did: follower.did,
                        platform: 'bluesky',
                        isFollower: true,
                        interactions: []
                    });
                }
            }
        } catch (e) {
            console.error('Error parsing Bluesky followers:', e.message);
        }
    }

    return { contacts, messages };
}
