import { readFileSync, existsSync, readdirSync } from 'fs';
import { join } from 'path';

/**
 * Parse Facebook data export (JSON format)
 */
export async function parseFacebook(dataDir) {
    const contacts = [];
    const messages = [];

    // Parse friends list
    const friendsFile = findJsonFile(dataDir, 'friends_and_followers', 'friends.json');
    if (friendsFile) {
        try {
            const data = JSON.parse(readFileSync(friendsFile, 'utf-8'));
            const friendsList = data.friends_v2 || data.friends || [];

            for (const friend of friendsList) {
                contacts.push({
                    name: friend.name || 'Unknown',
                    connectedDate: friend.timestamp ? new Date(friend.timestamp * 1000).toISOString() : null,
                    platform: 'facebook',
                    interactions: []
                });
            }
        } catch (e) {
            console.error('Error parsing Facebook friends:', e.message);
        }
    }

    // Parse messages
    const inboxDir = join(dataDir, 'messages', 'inbox');
    if (existsSync(inboxDir)) {
        const conversations = readdirSync(inboxDir);

        for (const convFolder of conversations) {
            const msgFile = join(inboxDir, convFolder, 'message_1.json');
            if (existsSync(msgFile)) {
                try {
                    const data = JSON.parse(readFileSync(msgFile, 'utf-8'));
                    const participants = data.participants?.map(p => p.name) || [];

                    // Find or create contact for each participant
                    for (const participant of participants) {
                        let contact = contacts.find(c => c.name === participant);
                        if (!contact) {
                            contact = {
                                name: participant,
                                platform: 'facebook',
                                interactions: []
                            };
                            contacts.push(contact);
                        }

                        // Add message interaction
                        const messageCount = data.messages?.length || 0;
                        const lastMessage = data.messages?.[0];

                        contact.interactions.push({
                            type: 'message',
                            count: messageCount,
                            lastDate: lastMessage?.timestamp_ms
                                ? new Date(lastMessage.timestamp_ms).toISOString()
                                : null,
                            hasDeepMessages: data.messages?.some(m =>
                                m.content && m.content.length > 200
                            )
                        });
                    }

                    messages.push({
                        participants,
                        messageCount: data.messages?.length || 0,
                        platform: 'facebook'
                    });
                } catch (e) {
                    // Skip invalid message files
                }
            }
        }
    }

    return { contacts, messages };
}

function findJsonFile(baseDir, ...pathParts) {
    const fullPath = join(baseDir, ...pathParts);
    return existsSync(fullPath) ? fullPath : null;
}
