import { parseLinkedIn } from './linkedin.js';
import { parseFacebook } from './facebook.js';
import { parseTwitter } from './twitter.js';
import { parseInstagram } from './instagram.js';
import { parseBluesky } from './bluesky.js';
import { parseThreads } from './threads.js';
import { existsSync } from 'fs';
import { join } from 'path';

/**
 * Load and merge data from all available platforms
 */
export async function loadAllData(dataDir) {
    const platforms = [];
    let allContacts = [];
    let allMessages = [];
    let followedCompanies = [];

    const parsers = [
        { name: 'linkedin', path: 'linkedin', parser: parseLinkedIn },
        { name: 'facebook', path: 'facebook', parser: parseFacebook },
        { name: 'twitter', path: 'twitter', parser: parseTwitter },
        { name: 'instagram', path: 'instagram', parser: parseInstagram },
        { name: 'bluesky', path: 'bluesky', parser: parseBluesky },
        { name: 'threads', path: 'threads', parser: parseThreads }
    ];

    for (const { name, path, parser } of parsers) {
        const platformDir = join(dataDir, path);

        if (existsSync(platformDir)) {
            try {
                console.log(`Parsing ${name} data...`);
                const data = await parser(platformDir);

                if (data.contacts.length > 0) {
                    platforms.push(name);
                    allContacts = allContacts.concat(data.contacts);
                    allMessages = allMessages.concat(data.messages || []);
                    console.log(`  - Found ${data.contacts.length} contacts`);
                }

                // Collect followed companies (primarily from LinkedIn)
                if (data.followedCompanies && data.followedCompanies.length > 0) {
                    followedCompanies = followedCompanies.concat(data.followedCompanies);
                    console.log(`  - Found ${data.followedCompanies.length} followed companies`);
                }
            } catch (error) {
                console.error(`Error parsing ${name}:`, error.message);
            }
        }
    }

    // Merge duplicate contacts across platforms
    const mergedContacts = mergeContacts(allContacts);

    return {
        contacts: mergedContacts,
        messages: allMessages,
        platforms,
        followedCompanies
    };
}


/**
 * Merge contacts that appear on multiple platforms
 */
function mergeContacts(contacts) {
    const contactMap = new Map();

    for (const contact of contacts) {
        // Create a key based on email or normalized name
        const key = contact.email?.toLowerCase() ||
            normalizeNameForMatching(contact.name);

        if (contactMap.has(key)) {
            // Merge with existing contact
            const existing = contactMap.get(key);
            existing.platforms = [...new Set([...existing.platforms, contact.platform])];
            existing.interactions = [...existing.interactions, ...contact.interactions];

            // Keep the most detailed info
            if (!existing.company && contact.company) existing.company = contact.company;
            if (!existing.title && contact.title) existing.title = contact.title;
            if (!existing.email && contact.email) existing.email = contact.email;
        } else {
            contactMap.set(key, {
                ...contact,
                platforms: [contact.platform],
                id: crypto.randomUUID()
            });
        }
    }

    return Array.from(contactMap.values());
}

/**
 * Normalize name for fuzzy matching
 */
function normalizeNameForMatching(name) {
    if (!name) return '';
    return name.toLowerCase()
        .replace(/[^a-z\s]/g, '')
        .replace(/\s+/g, ' ')
        .trim();
}
