import { readFileSync, existsSync, readdirSync } from 'fs';
import { join } from 'path';

/**
 * Parse Instagram data export (JSON format)
 */
export async function parseInstagram(dataDir) {
    const contacts = [];
    const messages = [];

    // Parse following
    const followingFile = join(dataDir, 'connections', 'followers_and_following', 'following.json');
    if (existsSync(followingFile)) {
        try {
            const data = JSON.parse(readFileSync(followingFile, 'utf-8'));
            const following = data.relationships_following || [];

            for (const item of following) {
                const stringData = item.string_list_data?.[0];
                contacts.push({
                    name: stringData?.value || 'Unknown',
                    handle: stringData?.value,
                    connectedDate: stringData?.timestamp
                        ? new Date(stringData.timestamp * 1000).toISOString()
                        : null,
                    platform: 'instagram',
                    interactions: []
                });
            }
        } catch (e) {
            console.error('Error parsing Instagram following:', e.message);
        }
    }

    // Parse followers
    const followersFile = join(dataDir, 'connections', 'followers_and_following', 'followers_1.json');
    if (existsSync(followersFile)) {
        try {
            const data = JSON.parse(readFileSync(followersFile, 'utf-8'));

            for (const item of data) {
                const stringData = item.string_list_data?.[0];
                const handle = stringData?.value;

                const existing = contacts.find(c => c.handle === handle);
                if (existing) {
                    existing.isMutual = true;
                } else {
                    contacts.push({
                        name: handle || 'Unknown',
                        handle,
                        platform: 'instagram',
                        isFollower: true,
                        interactions: []
                    });
                }
            }
        } catch (e) {
            console.error('Error parsing Instagram followers:', e.message);
        }
    }

    // Parse messages
    const inboxDir = join(dataDir, 'messages', 'inbox');
    if (existsSync(inboxDir)) {
        const conversations = readdirSync(inboxDir).filter(f =>
            !f.startsWith('.')
        );

        for (const convFolder of conversations) {
            const msgFile = join(inboxDir, convFolder, 'message_1.json');
            if (existsSync(msgFile)) {
                try {
                    const data = JSON.parse(readFileSync(msgFile, 'utf-8'));
                    const participants = data.participants?.map(p => p.name) || [];

                    for (const participant of participants) {
                        let contact = contacts.find(c =>
                            c.name === participant || c.handle === participant
                        );

                        if (!contact) {
                            contact = {
                                name: participant,
                                platform: 'instagram',
                                interactions: []
                            };
                            contacts.push(contact);
                        }

                        contact.interactions.push({
                            type: 'message',
                            count: data.messages?.length || 0,
                            lastDate: data.messages?.[0]?.timestamp_ms
                                ? new Date(data.messages[0].timestamp_ms).toISOString()
                                : null
                        });
                    }

                    messages.push({
                        participants,
                        messageCount: data.messages?.length || 0,
                        platform: 'instagram'
                    });
                } catch (e) {
                    // Skip invalid files
                }
            }
        }
    }

    return { contacts, messages };
}
