import { readFileSync, existsSync, readdirSync } from 'fs';
import { join } from 'path';
import { parse } from 'csv-parse/sync';

/**
 * Parse LinkedIn data export
 * LinkedIn exports data as CSV files in a ZIP
 */
export async function parseLinkedIn(dataDir) {
    const contacts = [];
    const messages = [];

    // Parse Connections.csv
    const connectionsFile = findFile(dataDir, 'Connections.csv');
    if (connectionsFile) {
        const content = readFileSync(connectionsFile, 'utf-8');
        // Skip the header row note that LinkedIn adds
        const csvContent = content.split('\n').slice(3).join('\n');

        try {
            const records = parse(csvContent, {
                columns: true,
                skip_empty_lines: true,
                trim: true
            });

            for (const record of records) {
                contacts.push({
                    name: `${record['First Name'] || ''} ${record['Last Name'] || ''}`.trim(),
                    email: record['Email Address'] || null,
                    company: record['Company'] || null,
                    title: record['Position'] || null,
                    connectedDate: parseDate(record['Connected On']),
                    platform: 'linkedin',
                    interactions: []
                });
            }
        } catch (e) {
            console.error('Error parsing Connections.csv:', e.message);
        }
    }

    // Parse messages.csv for interaction data
    const messagesFile = findFile(dataDir, 'messages.csv');
    if (messagesFile) {
        const content = readFileSync(messagesFile, 'utf-8');

        try {
            const records = parse(content, {
                columns: true,
                skip_empty_lines: true,
                trim: true,
                relax_column_count: true
            });

            // Group messages by conversation
            const conversationMap = new Map();

            for (const record of records) {
                const conversationId = record['CONVERSATION ID'];
                const from = record['FROM'];
                const date = record['DATE'];
                const messageContent = record['CONTENT'];

                if (!conversationMap.has(conversationId)) {
                    conversationMap.set(conversationId, []);
                }

                conversationMap.get(conversationId).push({
                    from,
                    date: parseDate(date),
                    content: messageContent,
                    isDeep: isDeepMessage(messageContent)
                });
            }

            // Attach messages to contacts
            for (const [convId, msgs] of conversationMap) {
                // Find the other person in the conversation
                const otherPerson = msgs.find(m => m.from && !m.from.includes('Your Name'))?.from;

                if (otherPerson) {
                    const contact = contacts.find(c =>
                        c.name.toLowerCase().includes(otherPerson.toLowerCase().split(' ')[0])
                    );

                    if (contact) {
                        contact.interactions.push({
                            type: 'message',
                            count: msgs.length,
                            lastDate: msgs[msgs.length - 1]?.date,
                            hasDeepMessages: msgs.some(m => m.isDeep),
                            conversationId: convId
                        });
                    }
                }

                messages.push({
                    conversationId: convId,
                    messages: msgs,
                    platform: 'linkedin'
                });
            }
        } catch (e) {
            console.error('Error parsing messages.csv:', e.message);
        }
    }

    // Parse endorsements
    const endorsementsReceived = findFile(dataDir, 'Endorsement_Received_Info.csv');
    if (endorsementsReceived) {
        try {
            const content = readFileSync(endorsementsReceived, 'utf-8');
            const records = parse(content, { columns: true, skip_empty_lines: true });

            for (const record of records) {
                const endorserName = record['Endorser First Name'] + ' ' + record['Endorser Last Name'];
                const contact = contacts.find(c =>
                    c.name.toLowerCase() === endorserName.toLowerCase()
                );

                if (contact) {
                    contact.interactions.push({
                        type: 'endorsement_received',
                        skill: record['Skill Name'],
                        date: parseDate(record['Endorsement Date'])
                    });
                }
            }
        } catch (e) {
            console.error('Error parsing endorsements:', e.message);
        }
    }

    // Parse recommendations
    const recommendationsReceived = findFile(dataDir, 'Recommendations_Received.csv');
    if (recommendationsReceived) {
        try {
            const content = readFileSync(recommendationsReceived, 'utf-8');
            const records = parse(content, { columns: true, skip_empty_lines: true });

            for (const record of records) {
                const recommenderName = record['First Name'] + ' ' + record['Last Name'];
                const contact = contacts.find(c =>
                    c.name.toLowerCase() === recommenderName.toLowerCase()
                );

                if (contact) {
                    contact.interactions.push({
                        type: 'recommendation_received',
                        text: record['Text'],
                        date: parseDate(record['Creation Date'])
                    });
                }
            }
        } catch (e) {
            console.error('Error parsing recommendations:', e.message);
        }
    }

    // Parse endorsements given (for reciprocity tracking)
    const endorsementsGiven = findFile(dataDir, 'Endorsement_Given_Info.csv');
    if (endorsementsGiven) {
        try {
            const content = readFileSync(endorsementsGiven, 'utf-8');
            const records = parse(content, { columns: true, skip_empty_lines: true });

            for (const record of records) {
                const endorseeName = record['Endorsee First Name'] + ' ' + record['Endorsee Last Name'];
                const contact = contacts.find(c =>
                    c.name.toLowerCase() === endorseeName.toLowerCase()
                );

                if (contact) {
                    contact.interactions.push({
                        type: 'endorsement_given',
                        skill: record['Skill Name'],
                        date: parseDate(record['Endorsement Date'])
                    });
                }
            }
        } catch (e) {
            console.error('Error parsing endorsements given:', e.message);
        }
    }

    // Parse invitations (who initiated connection)
    const invitationsFile = findFile(dataDir, 'Invitations.csv');
    if (invitationsFile) {
        try {
            const content = readFileSync(invitationsFile, 'utf-8');
            const records = parse(content, { columns: true, skip_empty_lines: true, relax_column_count: true });

            for (const record of records) {
                const direction = record['Direction'];
                const otherPerson = direction === 'OUTGOING' ? record['To'] : record['From'];

                const contact = contacts.find(c =>
                    c.name.toLowerCase() === otherPerson?.toLowerCase()
                );

                if (contact) {
                    contact.connectionInitiator = direction === 'OUTGOING' ? 'you' : 'them';
                    contact.invitationMessage = record['Message'] || null;
                    contact.invitationDate = parseDate(record['Sent At']);
                }
            }
        } catch (e) {
            console.error('Error parsing invitations:', e.message);
        }
    }

    // Parse company follows (target organizations)
    const companyFollowsFile = findFile(dataDir, 'Company Follows.csv');
    const followedCompanies = [];
    if (companyFollowsFile) {
        try {
            const content = readFileSync(companyFollowsFile, 'utf-8');
            const records = parse(content, { columns: true, skip_empty_lines: true });

            for (const record of records) {
                followedCompanies.push({
                    name: record['Organization'],
                    followedOn: parseDate(record['Followed On'])
                });
            }
        } catch (e) {
            console.error('Error parsing company follows:', e.message);
        }
    }

    return { contacts, messages, followedCompanies };
}

/**
 * Check if a message indicates deep conversation
 */
function isDeepMessage(content) {

    if (!content) return false;
    // Messages over 200 chars or with substantive keywords
    if (content.length > 200) return true;

    const deepIndicators = [
        'would love to', 'catch up', 'let me know', 'help you',
        'opportunity', 'collaborate', 'project', 'idea',
        'thoughts on', 'advice', 'recommend', 'introduce'
    ];

    return deepIndicators.some(indicator =>
        content.toLowerCase().includes(indicator)
    );
}

/**
 * Find a file in directory (case-insensitive)
 */
function findFile(dir, filename) {
    if (!existsSync(dir)) return null;

    const files = readdirSync(dir);
    const match = files.find(f => f.toLowerCase() === filename.toLowerCase());
    return match ? join(dir, match) : null;
}

/**
 * Parse various date formats
 */
function parseDate(dateStr) {
    if (!dateStr) return null;
    const date = new Date(dateStr);
    return isNaN(date.getTime()) ? null : date.toISOString();
}
