import { readFileSync, existsSync, readdirSync } from 'fs';
import { join } from 'path';

/**
 * Parse Threads data export (similar to Instagram format)
 */
export async function parseThreads(dataDir) {
    const contacts = [];
    const messages = [];

    // Parse following
    const followingFile = join(dataDir, 'connections', 'following.json');
    const altFollowingFile = join(dataDir, 'following.json');
    const followingPath = existsSync(followingFile) ? followingFile :
        existsSync(altFollowingFile) ? altFollowingFile : null;

    if (followingPath) {
        try {
            const data = JSON.parse(readFileSync(followingPath, 'utf-8'));
            const following = data.text_post_app_text_post_following || data.following || [];

            for (const item of following) {
                const stringData = item.string_list_data?.[0] || item;
                contacts.push({
                    name: stringData?.value || stringData?.username || 'Unknown',
                    handle: stringData?.value || stringData?.username,
                    connectedDate: stringData?.timestamp
                        ? new Date(stringData.timestamp * 1000).toISOString()
                        : null,
                    platform: 'threads',
                    interactions: []
                });
            }
        } catch (e) {
            console.error('Error parsing Threads following:', e.message);
        }
    }

    // Parse followers
    const followersFile = join(dataDir, 'connections', 'followers.json');
    const altFollowersFile = join(dataDir, 'followers.json');
    const followersPath = existsSync(followersFile) ? followersFile :
        existsSync(altFollowersFile) ? altFollowersFile : null;

    if (followersPath) {
        try {
            const data = JSON.parse(readFileSync(followersPath, 'utf-8'));
            const followers = data.text_post_app_text_post_followers || data.followers || data || [];

            for (const item of (Array.isArray(followers) ? followers : [])) {
                const stringData = item.string_list_data?.[0] || item;
                const handle = stringData?.value || stringData?.username;

                const existing = contacts.find(c => c.handle === handle);
                if (existing) {
                    existing.isMutual = true;
                } else {
                    contacts.push({
                        name: handle || 'Unknown',
                        handle,
                        platform: 'threads',
                        isFollower: true,
                        interactions: []
                    });
                }
            }
        } catch (e) {
            console.error('Error parsing Threads followers:', e.message);
        }
    }

    return { contacts, messages };
}
