import { readFileSync, existsSync, readdirSync } from 'fs';
import { join } from 'path';

/**
 * Parse Twitter/X data export (JSON format)
 */
export async function parseTwitter(dataDir) {
    const contacts = [];
    const messages = [];

    // Parse following list
    const followingFile = join(dataDir, 'data', 'following.js');
    if (existsSync(followingFile)) {
        try {
            let content = readFileSync(followingFile, 'utf-8');
            // Twitter exports as JS with variable assignment
            content = content.replace(/^window\.YTD\.following\.part0\s*=\s*/, '');
            const data = JSON.parse(content);

            for (const item of data) {
                const following = item.following;
                contacts.push({
                    name: following.userLink?.split('/').pop() || 'Unknown',
                    handle: following.userLink?.split('/').pop(),
                    platform: 'twitter',
                    interactions: []
                });
            }
        } catch (e) {
            console.error('Error parsing Twitter following:', e.message);
        }
    }

    // Parse followers
    const followersFile = join(dataDir, 'data', 'follower.js');
    if (existsSync(followersFile)) {
        try {
            let content = readFileSync(followersFile, 'utf-8');
            content = content.replace(/^window\.YTD\.follower\.part0\s*=\s*/, '');
            const data = JSON.parse(content);

            for (const item of data) {
                const follower = item.follower;
                const handle = follower.userLink?.split('/').pop();

                // Check if already in contacts (mutual follow)
                const existing = contacts.find(c => c.handle === handle);
                if (existing) {
                    existing.isMutual = true;
                } else {
                    contacts.push({
                        name: handle || 'Unknown',
                        handle,
                        platform: 'twitter',
                        isFollower: true,
                        interactions: []
                    });
                }
            }
        } catch (e) {
            console.error('Error parsing Twitter followers:', e.message);
        }
    }

    // Parse DMs
    const dmsFile = join(dataDir, 'data', 'direct-messages.js');
    if (existsSync(dmsFile)) {
        try {
            let content = readFileSync(dmsFile, 'utf-8');
            content = content.replace(/^window\.YTD\.direct_messages\.part0\s*=\s*/, '');
            const data = JSON.parse(content);

            for (const conv of data) {
                const dmConv = conv.dmConversation;
                const participantIds = dmConv.conversationId?.split('-') || [];

                messages.push({
                    conversationId: dmConv.conversationId,
                    messageCount: dmConv.messages?.length || 0,
                    platform: 'twitter'
                });
            }
        } catch (e) {
            console.error('Error parsing Twitter DMs:', e.message);
        }
    }

    return { contacts, messages };
}
